<?php

namespace App\Livewire\Admin\ProjectFiles;

use App\Models\{ProjectFile, Project};
use Livewire\{Component, WithPagination, Attributes\Layout, Attributes\Url, Attributes\Computed};
use Illuminate\Support\Facades\{Storage, Auth};

class ListFiles extends Component
{
    use WithPagination;

    #[Url(history: true)] public $search = "";
    #[Url] public $filterProject = "";
    #[Url] public $dateRange = "all";

    public $sortField = 'created_at';
    public $sortDirection = 'desc';
    public $selectedFiles = [];
    public $selectAll = false;

    protected $paginationTheme = 'bootstrap';

    public function updatedSearch() { $this->resetPage(); }
    public function updatedFilterProject() { $this->resetPage(); }
    public function updatedDateRange() { $this->resetPage(); }

    public function sortBy($field)
    {
        $this->sortDirection = ($this->sortField === $field && $this->sortDirection === 'asc') ? 'desc' : 'asc';
        $this->sortField = $field;
    }

    #[Computed]
    public function stats()
    {
        $bytes = ProjectFile::sum('size');
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $bytes > 1024; $i++) $bytes /= 1024;

        return [
            'total_count' => ProjectFile::count(),
            'formatted_size' => round($bytes, 2) . ' ' . $units[$i],
            'recent' => ProjectFile::where('created_at', '>=', now()->subDays(7))->count(),
        ];
    }

    #[Computed]
    public function projects() { return Project::orderBy('name')->get(['id', 'name']); }

    public function downloadFile($id)
    {
        $file = ProjectFile::findOrFail($id);
        if (!Storage::disk('public')->exists($file->path)) {
            $this->dispatch('alert', ['type' => 'error', 'message' => 'فایل یافت نشد.']);
            return;
        }
        return Storage::disk('public')->download($file->path, $file->name . '.' . $file->extension);
    }

    public function deleteFile($id)
    {
        if (!Auth::user()->isAdmin()) return;

        $file = ProjectFile::findOrFail($id);
        Storage::disk('public')->delete($file->path);
        $file->delete();
        $this->dispatch('alert', ['type' => 'warning', 'message' => 'فایل حذف شد.']);
    }

    public function deleteSelected()
    {
        if (!Auth::user()->isAdmin()) return;

        $files = ProjectFile::whereIn('id', $this->selectedFiles)->get();
        foreach ($files as $file) {
            Storage::disk('public')->delete($file->path);
            $file->delete();
        }
        $this->reset(['selectedFiles', 'selectAll']);
        $this->dispatch('alert', ['type' => 'success', 'message' => 'موارد انتخابی پاکسازی شدند.']);
    }

    #[Layout('admin.master')]
    public function render()
    {
        $projectFiles = ProjectFile::query()
            ->with(['project', 'section', 'user'])
            ->where(function($query) {
                $query->where('name', 'LIKE', '%' . $this->search . '%')
                    ->orWhereHas('project', fn($q) => $q->where('name', 'LIKE', '%' . $this->search . '%'))
                    ->orWhereHas('section', fn($q) => $q->where('name', 'LIKE', '%' . $this->search . '%'));
            })
            ->when($this->filterProject, fn($q) => $q->where('project_id', $this->filterProject))
            ->when($this->dateRange !== 'all', function($q) {
                return match($this->dateRange) {
                    'today' => $q->whereDate('created_at', now()),
                    'week' => $q->where('created_at', '>=', now()->startOfWeek()),
                    'month' => $q->where('created_at', '>=', now()->startOfMonth()),
                    default => $q
                };
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate(12);

        return view('livewire.admin.project-files.list-files', compact('projectFiles'));
    }
}
