<?php

namespace App\Livewire\Admin\ProjectLetter;

use App\Models\Letter;
use App\Models\Project;
use App\Models\LetterForm;
use App\Models\User;
use App\Models\Classification;
use App\Models\UserPosition;
use App\Models\Referral; // ✨ اضافه شده
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification; // ✨ اضافه شده
use Illuminate\Support\Collection;
use Carbon\Carbon;
use Morilog\Jalali\Jalalian;

class CreateLetter extends Component
{
    use WithFileUploads;

    public Project $project;
    public ?Letter $letter = null;

    // فیلدهای نامه
    public $title = '';
    public $content = '';
    public $from_who = '';
    public $to_who = '';
    public $from_where = ''; // ✨ FIX: متغیر جدید برای آدرس/واحد گیرنده
    public $classification_id = null;
    public $background_id = null;
    public $priority = 'normal';
    public $letter_date_shamsi = null;
    public $letter_date = null;
    public $attachments = [];
    public $type = 'internal';

    // فیلدهای امضاکننده و رونوشت
   // public $signer_id = null;
    public $signer_id;
    public $isSelfSigning = false; // پراپرتی جدید برای کنترل وضعیت
    public $signature_position = null;
    public $signerPositions = [];
    public $ccs = [], $cc_name = '', $cc_position = '';

    // وضعیت
    public $isEditing = false;
    public $autoSaveEnabled = true;
    public $lastAutoSavedAt = null;

    // متغیرهای ثابت
    public Collection $allUsers;
    public Collection $forms;
    public $confidential = false;

    // برای فیلتر کاربران رونوشت
    public $cc_user_id = null;
    public $cc_users_suggestions = [];

    protected $rules = [
        'title' => 'required|string|max:255',
        'content' => 'required|string|min:10',
        'from_who' => 'required|string|max:255',
        'from_where' => 'required|string|max:255',
        'priority' => 'required|in:low,normal,high,urgent',
        'attachments.*' => 'nullable|file|max:10240|mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png,zip,rar',
        'letter_date_shamsi' => 'required|string|max:10',
        'classification_id' => 'nullable|exists:classifications,id',
        'background_id' => 'required|exists:letter_forms,id',
        'type' => 'required|string|max:50',
        'signer_id' => 'required|exists:users,id',
        'signature_position' => 'required|string|max:100',
        'ccs' => 'array',
        'ccs.*.name' => 'required|string|max:100',
        'ccs.*.position' => 'nullable|string|max:100',
    ];



    public function mount(Project $project, $letterId = null)
    {
        $this->project = $project;
        $this->allUsers = User::where('is_active', true)->select('id', 'name')->get();
        $this->forms = $this->getAvailableLetterForms();

        // تنظیم پیش‌فرض‌ها
        $this->signer_id = Auth::id();

        if ($this->forms->count() === 1) {
            $this->background_id = $this->forms->first()->id;
        }

        $now = now();
        $this->letter_date = $now->format('Y-m-d');
        $this->letter_date_shamsi = Jalalian::fromCarbon($now)->format('Y/m/d');

        $this->cc_name = '';
        $this->cc_position = '';

        $isCloning = false;

        // ----------------------------------------------------
        // ✅ FIX 1: منطق خواندن داده‌های نامه کپی شده از سشن (Cloning)
        // ----------------------------------------------------
        if (session()->has('cloned_letter_data')) {
            $clonedData = session()->pull('cloned_letter_data');
            session()->pull('cloned_project_id');

            if ($clonedData) {
                $this->fill([
                    'title' => $clonedData['title'] . ' (کپی)',
                    'content' => $clonedData['content'] ?? '', // اطمینان از وجود محتوا
                    'from_who' => $clonedData['from_who'] ?? '',
                    'from_where' => $clonedData['from_where'] ?? '',
                    'classification_id' => $clonedData['classification_id'] ?? null,
                    'background_id' => $clonedData['background_id'],
                    'priority' => $clonedData['priority'] ?? 'normal',
                    'type' => $clonedData['type'] ?? 'internal',
                    'confidential' => $clonedData['confidential'] ?? false,

                    'signer_id' => $clonedData['signer_id'],
                    'signature_position' => $clonedData['signature_position'] ?? null,

                    'ccs' => $clonedData['ccs'] ?? [],
                ]);

                $this->isEditing = false;
                $this->autoSaveEnabled = true;
                $isCloning = true;

                $this->dispatch('show-toast', type: 'info', message: 'محتوای نامه کپی شده بارگزاری شد. شماره نامه جدید تولید خواهد شد.');

                // ✅ FIX 2: دیسپچ برای اطمینان از مقداردهی اولیه TinyMCE پس از mount
                $this->dispatch('content-loaded');
            }
        }
        // ----------------------------------------------------

        // اگر در حالت Cloning نیستیم، حالت ویرایش را بررسی می‌کنیم
        if (!$isCloning && $letterId) {
            $this->letter = Letter::where('project_id', $project->id)->findOrFail($letterId);
            $letterDate = $this->letter->letter_date ?? $this->letter->due_date;
            $this->fill([
                'title' => $this->letter->title,
                'content' => $this->letter->content,
                'from_who' => $this->letter->from_who,
                'from_where' => $this->letter->from_where,
                'classification_id' => $this->letter->classification_id,
                'background_id' => $this->letter->background_id,
                'priority' => $this->letter->priority,
                'type' => $this->letter->type,
                'signer_id' => $this->letter->user_id,
                'signature_position' => $this->letter->signature_position,
            ]);

            if ($letterDate) {
                $this->letter_date = $letterDate->format('Y-m-d');
                $this->letter_date_shamsi = Jalalian::fromCarbon($letterDate)->format('Y/m/d');
            }

            $this->ccs = $this->letter->copies->map(function ($copy) {
                return ['name' => $copy->name, 'position' => $copy->position];
            })->toArray();

            $this->isEditing = true;
            $this->autoSaveEnabled = false;

            // dispatch برای اطمینان از مقداردهی TinyMCE در حالت ویرایش
            $this->dispatch('content-loaded');
        }

        // اگر نه در حال ویرایش هستیم و نه در حال کپی، مطمئن می‌شویم که CC خالی باشد
        if (!$this->isEditing && !$isCloning) {
            $this->ccs = [];
        }

        $this->updateSignerPosition();
    }
    private function getAvailableLetterForms(): Collection
    {
        $userCompanyId = Auth::user()?->company_id;

        if (!$userCompanyId) {
            return Collection::make([]);
        }

        return LetterForm::where(function ($query) use ($userCompanyId) {
            $query->where('is_global', true)
                ->orWhereHas('companies', function ($q) use ($userCompanyId) {
                    $q->where('company_id', $userCompanyId);
                });
        })
            ->select('id', 'name')
            ->orderBy('name')
            ->get();
    }

    public function updatedLetterDateShamsi($value)
    {
        try {
            $carbonDate = Jalalian::fromFormat('Y/m/d', $value)->toCarbon();
            $this->letter_date = $carbonDate->format('Y-m-d');
            $this->validateOnly('letter_date_shamsi');
        } catch (\Exception $e) {
            $this->letter_date = null;
            Log::warning("Invalid Shamsi Date: " . $value);
        }
    }


    public function updatedSignerId($value)
    {
        // 1. بررسی اینکه آیا کاربر خودش امضاکننده است یا خیر
        $this->isSelfSigning = (int) $value === (int) auth()->id();

        // 2. اصلاح خطا: فراخوانی متدی که وجود دارد به جای متد ناموجود loadSignerPositions
        // ما از متد updateSignerPosition که در پایین تعریف کرده‌اید استفاده می‌کنیم
        $this->updateSignerPosition();
    }
    public function updateSignerPosition()
    {
        $this->signerPositions = [];
        $this->signature_position = null;

        if ($this->signer_id) {
            // دریافت تمام سمت‌های کاربر از جدول user_positions
            $positions = \App\Models\UserPosition::where('user_id', $this->signer_id)
                ->pluck('position')
                ->toArray();

            $this->signerPositions = array_unique(array_filter($positions));

            // اگر کاربر فقط یک سمت داشت، به صورت خودکار انتخاب شود
            if (count($this->signerPositions) === 1) {
                $this->signature_position = $this->signerPositions[0];
            }
        }
    }

    public function updatedCcName($value)
    {
        if (strlen($value) < 2) {
            $this->cc_users_suggestions = [];
            return;
        }

        $this->cc_users_suggestions = User::where('is_active', true)
            ->where('name', 'like', '%' . $value . '%')
            ->select('id', 'name')
            ->limit(5)
            ->get();
    }

    public function selectCcUser($userId, $userName)
    {
        $this->cc_name = $userName;
        $this->cc_user_id = $userId;
        $this->cc_users_suggestions = [];

        $positions = UserPosition::where('user_id', $userId)->pluck('position')->toArray();
        $uniquePositions = array_unique(array_filter($positions));

        if (count($uniquePositions) === 1) {
            $this->cc_position = $uniquePositions[0];
        } else {
            $this->cc_position = '';
        }
    }

    public function addCc()
    {
        try {
            // ✨ اصلاح: اعتبار سنجی قبل از افزودن و استفاده از trim
            $this->validate(['cc_name' => 'required|string|max:100', 'cc_position' => 'nullable|string|max:100']);
            if (trim($this->cc_name) === '') {
                $this->dispatch('show-toast', type: 'error', message: 'نام رونوشت نمی‌تواند خالی باشد.');
                return;
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-toast', type: 'error', message: 'نام رونوشت نمی‌تواند خالی باشد.');
            return;
        }

        $this->ccs[] = [
            'name' => trim($this->cc_name),
            'position' => trim($this->cc_position),
        ];
        $this->cc_name = '';
        $this->cc_position = '';
        $this->cc_user_id = null;
        $this->resetValidation(['cc_name', 'cc_position']);
    }

    public function removeCc($index)
    {
        unset($this->ccs[$index]);
        $this->ccs = array_values($this->ccs);
    }


    public function save()
    {
        // ۱. تشخیص وضعیت امضای خودکار
        $isSelfSigning = (int) $this->signer_id === (int) Auth::id();

        // ۲. اعتبار سنجی
        $this->validate();

        if (! $this->letter_date) {
            $this->dispatch('show-toast', type: 'error', message: 'تاریخ شمسی صحیح نیست. لطفا مجددا بررسی کنید.');
            return;
        }

        $data = $this->prepareData();
        $isNew = !$this->isEditing;
        $letter = $this->letter;
        $successMessage = '';

        \DB::beginTransaction();
        try {
            // تعیین وضعیت اولیه بر اساس نوع امضا
            $status = $isSelfSigning ? Letter::STATUS_APPROVED : Letter::STATUS_PENDING;

            if ($isSelfSigning) {
                $data['signed_at'] = now();
                // ذخیره مسیر تصویر امضای کاربر در رکورد نامه
                $data['signature_path'] = Auth::user()->sign;
            }

            // ۳. عملیات ذخیره یا به‌روزرسانی
            if ($this->isEditing) {
                $letter->update(array_merge($data, ['status' => $status]));
                $this->handleCopies($letter, true);
                $this->handleAttachments($letter);
            } else {
                $letter = $this->project->letters()->create(array_merge($data, [
                    'user_id'    => $this->signer_id,
                    'sender_id'  => Auth::id(),
                    'status'     => $status,
                    'code'       => $this->generateLetterCode(),
                    'unique_code' => (string) Str::uuid(),
                ]));
                $this->handleCopies($letter);
                $this->handleAttachments($letter);
            }

            // ۴. مدیریت گردش کار (Workflow)
            if ($isSelfSigning) {
                // ثبت رویداد امضای مستقیم در تاریخچه
                $this->logLetterEvent($letter, 'self_signed', 'نامه توسط نویسنده امضا و نهایی شد.');

                $successMessage = $isNew
                    ? 'نامه با موفقیت ثبت و **توسط شما امضا و نهایی شد**. کد نامه: ' . ($letter->code ?? 'N/A')
                    : 'نامه با موفقیت به‌روزرسانی و نهایی شد.';
            } else {
                // ایجاد ارجاع خودکار جهت امضا برای شخص دیگر
                $this->createSignerReferralAndNotify($letter);

                // ثبت رویداد ارجاع در تاریخچه
                $signerName = User::find($this->signer_id)->name ?? 'نامشخص';
                $this->logLetterEvent($letter, 'auto_referral', "نامه جهت امضا به صورت خودکار به «{$signerName}» ارجاع شد.");

                $successMessage = $isNew
                    ? 'نامه با موفقیت ثبت و برای امضا ارجاع داده شد. کد نامه: ' . ($letter->code ?? 'N/A')
                    : 'نامه با موفقیت به‌روزرسانی شد.';
            }

            \DB::commit();

        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error("Failed to save letter: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: 'خطا در ثبت نامه. (جزئیات در لاگ سیستم).');
            return;
        }

        // ۵. پاکسازی و خروجی
        $this->letter = $letter;
        $this->resetForm();
        $this->dispatch('close-modal');
        $this->dispatch('refresh-letters');
        $this->dispatch('show-toast', type: 'success', message: $successMessage);
        $this->dispatch('clear-tinymce-content');
    }

    protected function createSignerReferralAndNotify(Letter $letter): void
    {
        // ۱. جلوگیری از تکرار ارجاع امضا (اگر قبلاً در دیتابیس وجود داشت)
        $existingReferral = Referral::where('letter_id', $letter->id)
            ->where('user_id', $this->signer_id)
            ->where('type', 'approval')
            ->where('status', 'pending')
            ->first();

        if ($existingReferral) {
            // اگر ارجاع وجود دارد فقط نوتیفیکیشن مجدد ارسال شود (اختیاری)
            $this->sendSignatureNotification($letter);
            return;
        }

        // ۲. ایجاد ارجاع جدید در جدول referrals
        try {
            Referral::create([
                'letter_id'  => $letter->id,
                'user_id'    => $this->signer_id, // گیرنده امضا
                'created_by' => Auth::id(),      // نویسنده نامه
                'type'       => 'approval',      // نوع ارجاع: امضا
                'status'     => 'pending',
            ]);

            // ۳. ارسال نوتیفیکیشن (اصلاح پارامترها برای جلوگیری از خطا)
            $this->sendSignatureNotification($letter);

        } catch (\Exception $e) {
            Log::error("Failed to create signer referral for Letter ID {$letter->id}: " . $e->getMessage());
        }
    }
    protected function logLetterEvent(Letter $letter, string $type, string $description): void
    {
        if (class_exists(\App\Models\LetterEvent::class)) {
            \App\Models\LetterEvent::create([
                'letter_id'   => $letter->id,
                'user_id'     => Auth::id(),
                'type'        => $type,
                'description' => $description,
            ]);
        }
    }
    public function logSelfSignedEvent(Letter $letter): void
    {
        if (class_exists(\App\Models\LetterEvent::class)) {
            \App\Models\LetterEvent::create([
                'letter_id' => $letter->id,
                'user_id' => Auth::id(),
                'type' => 'self_signed',
                'description' => 'نامه به دلیل یکسان بودن نویسنده و امضاکننده، به‌صورت خودکار امضا و نهایی شد.',
            ]);
        }
    }

    private function sendSignatureNotification(Letter $letter): void
    {
        $signer = User::find($this->signer_id);

        // بررسی وجود کلاس نوتیفیکیشن ارجاع (LetterReferralSent)
        if ($signer && class_exists(\App\Notifications\LetterReferralSent::class)) {
            Notification::send($signer, new \App\Notifications\LetterReferralSent(
                $letter->id, // پارامتر اول: ID نامه
                Auth::id(),  // پارامتر دوم: ID فرستنده
                'امضا'        // پارامتر سوم: هدف ارجاع
            ));
        }
    }

    private function prepareData()
    {
        return [
            'title' => $this->title,
            'content' => $this->content,
            'from_who' => $this->from_who,
            'from_where' => $this->from_where,
            'classification_id' => $this->classification_id,
            'background_id' => $this->background_id,
            'priority' => $this->priority,
            'confidential' => $this->confidential, // ✨ استفاده از confidential
            'due_date' => $this->letter_date,
            'type' => $this->type,
            'user_id' => $this->signer_id,
            'signature_position' => $this->signature_position,
        ];
    }

    private function handleCopies(Letter $letter, $isEditing = false)
    {
        if ($isEditing) {
            $letter->copies()->delete();
        }

        foreach ($this->ccs as $copy) {
            $letter->copies()->create([
                'name' => $copy['name'],
                'position' => $copy['position'] ?? null,
            ]);
        }
    }

    private function handleAttachments($letter = null)
    {
        if (empty($this->attachments)) return;

        $targetLetter = $letter ?? $this->letter;
        if (!$targetLetter) return;

        foreach ($this->attachments as $file) {
            $path = $file->store('letters/' . $targetLetter->id, 'public');
            $targetLetter->attachments()->create([
                'file_path' => $path,
                'original_name' => $file->getClientOriginalName(),
            ]);
        }
        $this->attachments = [];
    }

    private function generateLetterCode()
    {
        $projectCode = $this->project->code ?: 'NOCODE';
        $datePart = now()->format('Ymd');
        $timePart = now()->format('His');
        return "LTR/{$projectCode}/{$datePart}/{$timePart}";
    }

    public function autoSave()
    {
        if (!$this->autoSaveEnabled || $this->isEditing) return;
        if (empty($this->title) && empty($this->content)) return;

        try {
            $this->validateOnly('title');
        } catch (\Throwable $th) {
            return;
        }

        $draft = $this->project->letters()
            ->where('user_id', Auth::id())
            ->where('status', Letter::STATUS_DRAFT)
            ->whereNull('signed_at')
            ->latest()
            ->first();

        $data = $this->prepareData();
        $data['status'] = Letter::STATUS_DRAFT;
        $data['user_id'] = $this->signer_id;
        $data['sender_id'] = Auth::id();

        try {
            if ($draft) {
                $data['code'] = $draft->code;
                $data['unique_code'] = $draft->unique_code;
                $draft->update($data);
            } else {
                $data['code'] = $this->generateLetterCode();
                $data['unique_code'] = Str::uuid();
                $draft = $this->project->letters()->create($data);
            }
        } catch (\Exception $e) {
            Log::warning("Autosave failed: " . $e->getMessage());
            return;
        }

        $this->lastAutoSavedAt = now()->format('H:i:s');
        $this->dispatch('show-toast', type: 'info', message: 'پیش‌نویس به‌طور خودکار ذخیره شد.');
    }

    public function previewLetter()
    {
        try {
            $this->validate([
                'title' => 'required',
                'content' => 'required',
                'from_who' => 'required',
                'from_where' => 'required',
                'letter_date_shamsi' => 'required|string',
                'background_id' => 'required|exists:letter_forms,id',
                'signer_id' => 'required|exists:users,id',
                'signature_position' => 'required|string|max:100',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-toast', type: 'error', message: 'برای پیش‌نمایش، فیلدهای اصلی نامه، سربرگ و امضا باید پر باشند.');
            return;
        }

        $htmlContent = $this->getPreviewHtml();
        $this->dispatch('open-preview-html', htmlContent: $htmlContent);
    }

    public function getPreviewHtml(): string
    {
        try {
            if (!$this->background_id || !$this->signer_id) {
                throw new \Exception("قالب سربرگ و امضاکننده باید انتخاب شوند.");
            }

            $backgroundForm = LetterForm::find($this->background_id);
            $signerUser = User::find($this->signer_id);

            if (!$backgroundForm || !$signerUser) {
                throw new \Exception("داده‌های لازم برای پیش‌نمایش (سربرگ یا امضاکننده) یافت نشد.");
            }

            $previewData = [
                'title' => $this->title,
                'content' => $this->content,
                'project' => $this->project,
                'background' => $backgroundForm,
                'priority' => $this->priority,
                'from_who' => $this->from_who,
                'from_where' => $this->from_where,
                'letter_date' => $this->letter_date_shamsi,
                'priorityLabel' => $this->priorityLabel(),
                'ccs' => $this->ccs,
                'signer_name' => $signerUser->name,
                'signature_position' => $this->signature_position,
                'layoutSections' => ['header' => false, 'footer' => false, 'title' => true, 'metadata' => true, 'content' => true, 'attachments' => true, 'signature' => true],
            ];

            // 💡 اصلاح: ارسال آرایه داده‌ها زیر کلید 'data'
            return view('components.letter-preview-content', ['data' => $previewData])->render();
            Log::info('LETTER PREVIEW LENGTH: ' . strlen($html));

        } catch (\Exception $e) {
            Log::error("Preview generation failed: " . $e->getMessage());

            return "<div style='color: #dc3545; text-align: center; padding: 20px; background: #ffe3e6; border: 1px solid #dc3545; border-radius: 5px;'>
                    <strong style='font-size: 1.1em;'>خطا در تولید پیش‌نمایش:</strong>
                    <p style='margin-top: 10px;'>لطفاً از انتخاب قالب سربرگ و امضاکننده اطمینان حاصل کنید. جزئیات: " . e($e->getMessage()) . "</p>
                </div>";
        }
    }

    public function removeAttachment($index)
    {
        if (isset($this->attachments[$index])) {
            if (method_exists($this->attachments[$index], 'delete')) {
                $this->attachments[$index]->delete();
            }
            unset($this->attachments[$index]);
            $this->attachments = array_values($this->attachments);
            $this->dispatch('show-toast', type: 'info', message: 'پیوست موقت حذف شد.');
        }
    }

    private function resetForm()
    {
        $this->reset([
            'title', 'content', 'from_who', 'from_where', 'classification_id', 'background_id',
            'priority', 'letter_date', 'letter_date_shamsi', 'attachments',
            'cc_name', 'cc_position', 'ccs', 'signature_position', 'cc_user_id', 'cc_users_suggestions', 'confidential'
        ]);
        $this->letter = null;
        $this->isEditing = false;
        $this->autoSaveEnabled = true;
        $this->signer_id = Auth::id();
        $this->updateSignerPosition();
        $now = now();
        $this->letter_date = $now->format('Y-m-d');
        $this->letter_date_shamsi = Jalalian::fromCarbon($now)->format('Y/m/d');
        $this->resetErrorBag();
    }

    public function priorityLabel()
    {
        $labels = [
            'low' => 'کم',
            'normal' => 'عادی',
            'high' => 'بالا',
            'urgent' => 'فوری'
        ];
        return $labels[$this->priority] ?? 'نامشخص';
    }

    public function render()
    {
        $classifications = Classification::all();

        return view('livewire.admin.project-letter.create-letter', [
            'classifications' => $classifications,
            'allUsers' => $this->allUsers,
            'forms' => $this->forms,
        ])->layout('admin.master');
    }
}
