<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Project;
use App\Models\ProjectType;
use Livewire\Attributes\{Layout, Rule, Computed};
use Livewire\Component;
use Illuminate\Support\Facades\DB;

class CreateProject extends Component
{
    #[Rule('required|min:3|max:255|unique:projects,name', as: 'نام پروژه')]
    public $name = '';

    #[Rule('required|alpha_dash|min:3|unique:projects,code', as: 'کدینگ پروژه')]
    public $code = '';

    #[Rule('required|exists:project_types,id', as: 'نوع دسته‌بندی')]
    public $project_type_id = '';

    #[Computed]
    public function projects()
    {
        return Project::with('projectTypes')->latest()->take(6)->get();
    }

    #[Computed]
    public function projectTypes()
    {
        return ProjectType::orderBy('name')->get();
    }

    public function saveProject()
    {
        $this->validate();

        try {
            DB::transaction(function () {
                $project = Project::create([
                    'name' => $this->name,
                    'code' => strtoupper($this->code),
                    'is_active' => true,
                ]);

                if ($this->project_type_id) {
                    $project->projectTypes()->attach($this->project_type_id);
                }
            });

            $this->reset(['name', 'code', 'project_type_id']);
            $this->dispatch('alert', ['type' => 'success', 'message' => 'پروژه با هویت دیجیتال جدید ثبت شد.']);

        } catch (\Exception $e) {
            $this->dispatch('alert', ['type' => 'error', 'message' => 'خطا در برقراری ارتباط با سرور.']);
        }
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.projects.create-project');
    }
}
