<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Project;
use App\Models\User;
use App\Models\Role;
use App\Models\Company;
use App\Models\Department;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Layout;
use Livewire\Component;

class EditProjectMembers extends Component
{
    public Project $project;
    public $members;
    public $roles;
    public $allRolesArr;

    public $allUsers;
    public $companies;
    public $departments;

    public $selectedCompanyId = '';
    public $selectedDepartmentId = '';

    public string $search = '';
    public ?string $filterRoleId = '';
    public bool $showModal = false;
    public bool $editing = false;
    public $selectedUserId = null;
    public $selectedRoleId = null;
    public string $user_search = '';

    protected $queryString = [
        'search' => ['except' => '', 'as' => 'q'],
        'filterRoleId' => ['except' => '', 'as' => 'role'],
    ];

    protected $rules = [
        'selectedRoleId' => 'required|exists:roles,id'
    ];

    public function mount(Project $project)
    {
        // 🔒 دسترسی: فقط ادمین یا اعضای پروژه (user_roles)
        $user = Auth::user();
        if (!$user) {
            abort(403);
        }

        if (!$user->isAdmin() && !$user->projects()->where('project_id', $project->id)->exists()) {
            // 💡 اختیاری: اگر می‌خواهید کارمندان شرکت‌های ذینفع (کارفرما/پیمانکار/مشاور) هم دسترسی داشته باشند:
            // if (!$user->company || !$project->parties()->where('company_id', $user->company->id)->exists()) {
            abort(403, 'شما به این پروژه دسترسی ندارید.');
            // }
        }

        $this->project = $project;
        $this->roles = Role::where('is_project_role', true)->get();
        $this->companies = Company::select('id', 'name')->get();

        $this->allRolesArr = $this->roles->keyBy('id')->map(fn($r) => [
            'id' => $r->id,
            'name' => $r->name,
            'label' => $r->label ?? $r->name
        ])->toArray();

        $this->refreshMembers();
        $this->refreshAllUsers();
        $this->loadDepartments(false);
    }

    public function loadDepartments(bool $refreshUsers = true)
    {
        $query = Department::select('id', 'name');

        if ($this->selectedCompanyId) {
            $query->where('company_id', $this->selectedCompanyId);
        } else {
            $query->where('id', 0);
        }
        $this->departments = $query->orderBy('name')->get();

        $this->selectedDepartmentId = '';

        if ($refreshUsers) {
            $this->refreshAllUsers();
        }
    }

    public function updatedSearch() { $this->refreshMembers(); }
    public function updatedFilterRoleId() { $this->refreshMembers(); }
    public function updatedUserSearch() { $this->refreshAllUsers(); }
    public function updatedSelectedCompanyId() { $this->loadDepartments(); }
    public function updatedSelectedDepartmentId() { $this->loadDepartments(); }

    public function refreshMembers()
    {
        $query = $this->project->members()
            ->withPivot('role_id', 'status')
            ->with(['department.company:id,name', 'department:id,name']);

        if ($this->search) {
            $query->where(function($q) {
                $q->where('users.name', 'like', "%{$this->search}%")
                    ->orWhere('users.email', 'like', "%{$this->search}%");
            });
        }
        if ($this->filterRoleId) {
            $query->wherePivot('role_id', $this->filterRoleId);
        }

        $query->orderBy(DB::raw('LOWER(users.name)'));

        $this->members = $query->get();
    }

    public function refreshAllUsers()
    {
        $currentIds = $this->project->members->pluck('id')->toArray();
        $query = User::whereNotIn('id', $currentIds)
            ->with(['department.company:id,name', 'department:id,name']);

        if ($this->user_search) {
            $query->where(function($q) {
                $q->where('name', 'like', "%{$this->user_search}%")
                    ->orWhere('email', 'like', "%{$this->user_search}%");
            });
        }
        if ($this->selectedCompanyId) {
            $query->whereHas('department', function($q) {
                $q->where('company_id', $this->selectedCompanyId);
            });
        }
        if ($this->selectedDepartmentId) {
            $query->where('department_id', $this->selectedDepartmentId);
        }

        $this->allUsers = $query->limit(50)->get();
    }

    public function showAddForm()
    {
        $this->editing = false;
        $this->reset(['selectedUserId', 'selectedRoleId', 'user_search', 'selectedCompanyId', 'selectedDepartmentId']);
        $this->loadDepartments();
        $this->refreshAllUsers();
        $this->showModal = true;
    }

    public function editMember(int $userId)
    {
        // 🔒 چک مجدد در action (برای جلوگیری از Postman abuse)
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project->id)->exists())) {
            abort(403);
        }

        $member = $this->project->members()->where('user_id', $userId)->first();
        if ($member) {
            $this->editing = true;
            $this->showModal = true;
            $this->selectedUserId = $userId;
            $this->selectedRoleId = $member->pivot->role_id ?? null;
        }
    }

    public function saveMember()
    {
        // 🔒 چک مجدد در action
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project->id)->exists())) {
            abort(403);
        }

        $userIdToSave = $this->selectedUserId;
        $rulesToValidate = $this->rules;

        if (!$this->editing) {
            $rulesToValidate['selectedUserId'] = 'required|exists:users,id';
        }

        $this->validate($rulesToValidate);

        $this->project->members()->syncWithoutDetaching([
            $userIdToSave => ['role_id' => $this->selectedRoleId, 'status' => 'active']
        ]);

        session()->flash('success', $this->editing ? 'نقش عضو به‌روزرسانی شد.' : 'عضو جدید با موفقیت اضافه شد.');

        $this->showModal = false;
        $this->refreshMembers();
        $this->refreshAllUsers();
        $this->reset(['selectedUserId', 'selectedRoleId', 'editing', 'user_search', 'selectedCompanyId', 'selectedDepartmentId']);
    }

    public function removeUser(int $userId)
    {
        // 🔒 چک مجدد در action
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project->id)->exists())) {
            abort(403);
        }

        $this->project->members()->detach($userId);
        session()->flash('success', 'کاربر از پروژه حذف شد.');
        $this->refreshMembers();
        $this->refreshAllUsers();
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->reset(['selectedUserId', 'selectedRoleId', 'editing', 'user_search', 'selectedCompanyId', 'selectedDepartmentId']);
    }

    #[Layout('admin.master')]
    public function render()
    {
        $groupedUsers = $this->allUsers->groupBy(fn($u) => $u->department->company->name ?? 'بدون شرکت')
            ->map(fn($companyGroup) => $companyGroup->groupBy(fn($u) => $u->department->name ?? 'بدون دپارتمان'));

        return view('livewire.admin.projects.edit-project-members', [
            'members' => $this->members,
            'roles' => $this->roles,
            'companies' => $this->companies,
            'departments' => $this->departments,
            'allRolesArr' => $this->allRolesArr,
            'allUsers' => $this->allUsers,
            'groupedUsers' => $groupedUsers,
        ]);
    }
}
