<?php

namespace App\Livewire\Admin\Projects\Financial;

use App\Models\Project;
use App\Models\ProjectFinancialDetail;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('admin.master')]
class FinancialDetailCreate extends Component
{
    public $project_id;
    public $project;

    // فیلدهای نمایشی (با کاما)
    public $display_contract_value;
    public $display_exchange_rate;

    // فیلدهای ذخیره‌شونده (عدد خام)
    public $contract_value;
    public $currency_unit = 'IRR';
    public $exchange_rate;

    protected $exchangeRates = [
        'USD' => 580000,
        'EUR' => 620000,
        'GBP' => 730000,
        'AED' => 158000,
        'CAD' => 420000,
        'JPY' => 3800,
        'TRY' => 18000,
        'CNY' => 80000,
    ];

    protected $rules = [
        'project_id'      => 'required|exists:projects,id',
        'contract_value'  => 'nullable|numeric|min:0',
        'currency_unit'   => 'required|string|in:IRR,USD,EUR,GBP,AED,CAD,JPY,TRY,CNY',
        'exchange_rate'   => 'nullable|numeric|min:0',
    ];

    public function mount($projectId)
    {
        $this->project_id = $projectId;
        $this->project = Project::findOrFail($projectId);

        if ($finance = ProjectFinancialDetail::where('project_id', $projectId)->first()) {
            $this->contract_value = $finance->contract_value;
            $this->currency_unit  = $finance->currency_unit;
            $this->exchange_rate  = $finance->exchange_rate;
        }

        $this->syncDisplayValues();
    }

    public function updated($property)
    {
        if (in_array($property, ['contract_value', 'exchange_rate'])) {
            $this->syncDisplayValues();
        }

        if ($property === 'currency_unit') {
            if ($this->currency_unit !== 'IRR' && !$this->exchange_rate) {
                $this->exchange_rate = $this->exchangeRates[$this->currency_unit] ?? null;
            } elseif ($this->currency_unit === 'IRR') {
                $this->exchange_rate = null;
            }
            $this->syncDisplayValues();
        }
    }

    protected function syncDisplayValues()
    {
        $this->display_contract_value = $this->formatNumber($this->contract_value);
        $this->display_exchange_rate = $this->formatNumber($this->exchange_rate);
    }

    protected function formatNumber($value)
    {
        if ($value === null || $value === '') return '';
        return number_format((float) $value, 0, '.', ',');
    }

    protected function parseFormattedNumber($formatted)
    {
        if (!$formatted) return null;
        return (float) str_replace(',', '', $formatted);
    }

    public function saveFinancialDetail()
    {
        // تبدیل فیلدهای نمایشی به عدد خام قبل از اعتبارسنجی
        $this->contract_value = $this->parseFormattedNumber($this->display_contract_value);
        $this->exchange_rate = $this->parseFormattedNumber($this->display_exchange_rate);

        // ذخیره موقت برای نمایش مجدد در فرم در صورت خطا
        $temp_contract = $this->display_contract_value;
        $temp_exchange = $this->display_exchange_rate;

        $this->validate();

        ProjectFinancialDetail::updateOrCreate(
            ['project_id' => $this->project_id],
            [
                'contract_value' => $this->contract_value,
                'currency_unit'  => $this->currency_unit,
                'exchange_rate'  => $this->exchange_rate,
            ]
        );

        session()->flash('success', 'اطلاعات مالی پروژه با موفقیت ذخیره شد.');
    }

    public function getRialValueProperty()
    {
        if ($this->currency_unit === 'IRR') {
            return $this->contract_value;
        }
        if ($this->contract_value && $this->exchange_rate) {
            return $this->contract_value * $this->exchange_rate;
        }
        return null;
    }

    public function getCurrencyIconProperty()
    {
        $icons = [
            'IRR' => '﷼',
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'AED' => 'د.إ',
            'CAD' => 'C$',
            'JPY' => '¥',
            'TRY' => '₺',
            'CNY' => '¥',
        ];
        return $icons[$this->currency_unit] ?? '?';
    }

    public function render()
    {
        return view('livewire.admin.projects.financial.financial-detail-create');
    }
}
