<?php

namespace App\Livewire\Admin\Projects\Financial;

use App\Models\Project;
use App\Models\ProjectContract;
use App\Models\ProjectContractAddendum;
use App\Models\ProjectContractDocument;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;

#[Layout('admin.master')]
class ProjectContractManager extends Component
{
    use WithFileUploads;

    // === پروژه ===
    public $project_id;
    public $project;

    // === قرارداد اصلی ===
    public $contract_id;
    public $title;
    public $number;
    public $description;
    public $base_amount;
    public $currency_unit = 'IRR'; // ✅ قابل انتخاب
    public $exchange_rate;
    public $start_date;
    public $end_date;
    public $signed_at;
    public $status_id;
    public $is_main = true;
    public $display_base_amount;

    // === الحاقیه ===
    public $addendum_id;
    public $addendum_title;
    public $addendum_description;
    public $addendum_amount;
    public $addendum_currency_unit = 'IRR';
    public $addendum_exchange_rate;
    public $addendum_effective_date;
    public $addendum_approved_at;
    public $display_addendum_amount;

    // === اسناد ===
    public $document_file;
    public $document_title;

    // === قوانین اعتبارسنجی ===
    protected $rules = [
        'title' => 'required|string|max:255',
        'base_amount' => 'required|numeric|min:0',
        'currency_unit' => 'required|string|in:IRR,USD,EUR,GBP,AED,CAD,JPY,TRY,CNY',
        'start_date' => 'nullable|date',
        'end_date' => 'nullable|date|after_or_equal:start_date',
        'signed_at' => 'nullable|date',
    ];

    protected $addendumRules = [
        'addendum_title' => 'required|string|max:255',
        'addendum_amount' => 'required|numeric|min:0',
        'addendum_currency_unit' => 'required|string|in:IRR,USD,EUR,GBP,AED,CAD,JPY,TRY,CNY',
        'addendum_effective_date' => 'nullable|date',
        'addendum_approved_at' => 'nullable|date',
    ];

    // === متد اولیه ===
    public function mount($projectId)
    {
        $this->project_id = $projectId;
        $this->project = Project::with('mainContract.addenda')->findOrFail($projectId);

        if ($this->project->mainContract) {
            $this->loadContractData($this->project->mainContract);
        } else {
            $this->title = "قرارداد اصلی پروژه: {$this->project->name}";
        }
    }

    // === بارگذاری داده‌های قرارداد ===
    protected function loadContractData($contract)
    {
        $this->contract_id = $contract->id;
        $this->title = $contract->title;
        $this->number = $contract->number;
        $this->description = $contract->description;
        $this->base_amount = $contract->base_amount;
        $this->currency_unit = $contract->currency_unit;
        $this->exchange_rate = $contract->exchange_rate;
        $this->start_date = $contract->start_date?->format('Y-m-d');
        $this->end_date = $contract->end_date?->format('Y-m-d');
        $this->signed_at = $contract->signed_at?->format('Y-m-d');
        $this->status_id = $contract->status_id;
        $this->is_main = $contract->is_main;

        $this->syncDisplayValues();
    }

    // === همگام‌سازی مقادیر نمایشی ===
    public function updated($property)
    {
        if ($property === 'base_amount') {
            $this->display_base_amount = $this->formatNumber($this->base_amount);
        }

        if ($property === 'addendum_amount') {
            $this->display_addendum_amount = $this->formatNumber($this->addendum_amount);
        }

        if ($property === 'currency_unit' && $this->currency_unit === 'IRR') {
            $this->exchange_rate = null;
        }

        if ($property === 'addendum_currency_unit' && $this->addendum_currency_unit === 'IRR') {
            $this->addendum_exchange_rate = null;
        }
    }

    // === به‌روزرسانی مقادیر نمایشی ===
    public function updatedDisplayBaseAmount($value)
    {
        $this->base_amount = $this->parseFormattedNumber($value);
    }

    public function updatedDisplayAddendumAmount($value)
    {
        $this->addendum_amount = $this->parseFormattedNumber($value);
    }

    // === فرمت‌دهی اعداد ===
    protected function syncDisplayValues()
    {
        $this->display_base_amount = $this->formatNumber($this->base_amount);
        $this->display_addendum_amount = $this->formatNumber($this->addendum_amount);
    }

    protected function formatNumber($value)
    {
        if ($value === null || $value === '' || $value == 0) return '';
        return number_format((float) $value, 0, '.', ',');
    }

    protected function parseFormattedNumber($formatted)
    {
        if (!$formatted) return 0;
        return (float) str_replace(',', '', $formatted);
    }

    // === ذخیره قرارداد ===
    public function saveContract()
    {
        $this->validate();

        // تبدیل تاریخ‌های خالی به null
        $startDate = !empty($this->start_date) ? $this->start_date : null;
        $endDate = !empty($this->end_date) ? $this->end_date : null;
        $signedAt = !empty($this->signed_at) ? $this->signed_at : null;
        $exchangeRate = !empty($this->exchange_rate) ? $this->exchange_rate : null;

        ProjectContract::updateOrCreate(
            ['id' => $this->contract_id],
            [
                'project_id' => $this->project_id,
                'title' => $this->title,
                'number' => $this->number,
                'description' => $this->description,
                'base_amount' => $this->base_amount,
                'currency_unit' => $this->currency_unit,
                'exchange_rate' => $exchangeRate,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'signed_at' => $signedAt,
                'status_id' => $this->status_id,
                'is_main' => $this->is_main,
                'contract_type' => 'main',
            ]
        );

        session()->flash('success', 'قرارداد با موفقیت ذخیره شد.');
        $this->dispatch('contract-saved');
    }

    // === باز کردن مودال الحاقیه ===
    public function openAddendumModal($addendumId = null)
    {
        $this->resetAddendumFields();

        if ($addendumId) {
            $addendum = ProjectContractAddendum::findOrFail($addendumId);
            $this->addendum_id = $addendum->id;
            $this->addendum_title = $addendum->title;
            $this->addendum_description = $addendum->description;
            $this->addendum_amount = $addendum->amount;
            $this->addendum_currency_unit = $addendum->currency_unit;
            $this->addendum_exchange_rate = $addendum->exchange_rate;
            $this->addendum_effective_date = $addendum->effective_date?->format('Y-m-d');
            $this->addendum_approved_at = $addendum->approved_at?->format('Y-m-d');
        } else {
            $this->addendum_currency_unit = $this->currency_unit; // ✅ پیش‌فرض واحد قرارداد اصلی
        }

        $this->syncDisplayValues();
        $this->dispatch('open-addendum-modal');
    }

    // === ذخیره الحاقیه ===
    public function saveAddendum()
    {
        $this->validate($this->addendumRules);

        if (!$this->project->mainContract) {
            session()->flash('error', 'ابتدا قرارداد اصلی را ذخیره کنید.');
            return;
        }

        // ✅ تبدیل تاریخ‌های خالی به null
        $effectiveDate = !empty($this->addendum_effective_date) ? $this->addendum_effective_date : null;
        $approvedAt = !empty($this->addendum_approved_at) ? $this->addendum_approved_at : null;
        $exchangeRate = !empty($this->addendum_exchange_rate) ? $this->addendum_exchange_rate : null;

        if ($this->addendum_id) {
            $addendum = ProjectContractAddendum::findOrFail($this->addendum_id);
            $addendum->update([
                'title' => $this->addendum_title,
                'description' => $this->addendum_description,
                'amount' => $this->addendum_amount,
                'currency_unit' => $this->addendum_currency_unit,
                'exchange_rate' => $exchangeRate,
                'effective_date' => $effectiveDate,
                'approved_at' => $approvedAt,
            ]);
            session()->flash('success', 'الحاقیه با موفقیت به‌روزرسانی شد.');
        } else {
            ProjectContractAddendum::create([
                'project_contract_id' => $this->project->mainContract->id,
                'title' => $this->addendum_title,
                'description' => $this->addendum_description,
                'amount' => $this->addendum_amount,
                'currency_unit' => $this->addendum_currency_unit,
                'exchange_rate' => $exchangeRate,
                'effective_date' => $effectiveDate,
                'approved_at' => $approvedAt,
            ]);
            session()->flash('success', 'الحاقیه جدید با موفقیت اضافه شد.');
        }

        $this->resetAddendumFields();
        $this->dispatch('addendum-saved');
    }

    // === حذف الحاقیه ===
    public function deleteAddendum($id)
    {
        ProjectContractAddendum::destroy($id);
        session()->flash('success', 'الحاقیه با موفقیت حذف شد.');
        $this->dispatch('addendum-deleted');
    }

    // === ریست فیلدهای الحاقیه ===
    public function resetAddendumFields()
    {
        $this->addendum_id = null;
        $this->addendum_title = '';
        $this->addendum_description = '';
        $this->addendum_amount = 0;
        $this->addendum_currency_unit = $this->currency_unit; // ✅ پیش‌فرض واحد قرارداد اصلی
        $this->addendum_exchange_rate = null;
        $this->addendum_effective_date = '';
        $this->addendum_approved_at = '';
        $this->display_addendum_amount = '';
    }

    // === آپلود سند ===
    public function uploadDocument()
    {
        $this->validate([
            'document_file' => 'required|file|mimes:pdf,doc,docx|max:10240',
            'document_title' => 'required|string|max:255',
        ]);

        if (!$this->project->mainContract) {
            session()->flash('error', 'ابتدا قرارداد اصلی را ذخیره کنید.');
            return;
        }

        $path = $this->document_file->store('contracts/' . $this->project_id, 'public');

        ProjectContractDocument::create([
            'project_contract_id' => $this->project->mainContract->id,
            'uploaded_by' => auth()->id(),
            'title' => $this->document_title,
            'file_path' => $path,
        ]);

        session()->flash('success', 'سند قرارداد با موفقیت آپلود شد.');
        $this->document_file = null;
        $this->document_title = '';
    }

    // === دانلود سند ===
    public function downloadDocument($id)
    {
        $document = ProjectContractDocument::findOrFail($id);
        $filePath = storage_path('app/public/' . $document->file_path);
        return response()->download($filePath, $document->title);
    }

    // === حذف سند ===
    public function deleteDocument($id)
    {
        $document = ProjectContractDocument::findOrFail($id);
        Storage::disk('public')->delete($document->file_path);
        $document->delete();
        session()->flash('success', 'سند با موفقیت حذف شد.');
        $this->dispatch('document-deleted');
    }

    // === آیکون واحد پولی ===
    public function getCurrencyIconProperty()
    {
        return $this->getCurrencyIcon($this->currency_unit);
    }

    public function getCurrencyIcon($currency)
    {
        $icons = [
            'IRR' => '﷼',
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'AED' => 'د.إ',
            'CAD' => 'C$',
            'JPY' => '¥',
            'TRY' => '₺',
            'CNY' => '¥',
        ];
        return $icons[$currency] ?? '?';
    }

    // === رندر ویو ===
    public function render()
    {
        $documents = $this->project->mainContract
            ? ProjectContractDocument::where('project_contract_id', $this->project->mainContract->id)->get()
            : collect();

        return view('livewire.admin.projects.financial.project-contract-manager', [
            'documents' => $documents,
        ]);
    }
}
