<?php

namespace App\Livewire\Admin\Projects\Insurance;

use App\Models\ProjectInsurance;
use App\Models\Company;
use App\Models\Project;
use App\Models\InsuranceAttachment;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Carbon\Carbon;

class InsuranceDetailCreate extends Component
{
    use WithFileUploads;

    public $projectId;
    public $insuranceId = null;

    public $company_id;
    public $start_date;
    public $end_date;
    public $execution_duration;
    public $companies;

    public $newAttachments = [];
    public $currentAttachments = [];
    public $deletedAttachmentIds = [];

    public $projectName;
    public $projectCode;
    public $editing = false;

    protected $rules = [
        'company_id' => 'required|exists:companies,id',
        'start_date' => 'required|date',
        'end_date' => 'nullable|date|after:start_date',
        'execution_duration' => 'nullable|integer|min:1',
        'newAttachments' => 'nullable|array|max:5',
        'newAttachments.*' => 'file|mimes:pdf,jpg,jpeg,png|max:2048',
    ];

    public function mount($projectId)
    {
        // 🔒 امنیت: فقط ادمین یا اعضای پروژه (user_roles)
        $user = Auth::user();
        $project = Project::findOrFail($projectId);

        abort_unless($user, 403);
        abort_unless(
            $user->isAdmin() || $user->projects()->where('project_id', $project->id)->exists(),
            403,
            'شما به این پروژه دسترسی ندارید.'
        );

        $this->projectId = $projectId;
        $this->companies = Company::select('id', 'name')->orderBy('name')->get();

        $this->projectName = $project->name;
        $this->projectCode = $project->code;

        $this->resetForm();
    }

    public function resetForm()
    {
        $this->reset([
            'insuranceId', 'company_id', 'start_date', 'end_date',
            'execution_duration', 'editing', 'newAttachments',
            'currentAttachments', 'deletedAttachmentIds'
        ]);

        $this->start_date = '';
        $this->end_date = '';
        $this->execution_duration = null;
    }

    public function editInsurance(int $id)
    {
        // 🔒 چک مجدد در action
        $user = Auth::user();
        abort_unless($user?->isAdmin() || $user->projects()->where('project_id', $this->projectId)->exists(), 403);

        $insurance = ProjectInsurance::with('attachments')->findOrFail($id);

        $this->insuranceId = $insurance->id;
        $this->company_id = $insurance->company_id;
        $this->start_date = $insurance->start_date ? Carbon::parse($insurance->start_date)->toDateString() : '';
        $this->end_date = $insurance->end_date ? Carbon::parse($insurance->end_date)->toDateString() : '';
        $this->execution_duration = $insurance->execution_duration;
        $this->currentAttachments = $insurance->attachments;
        $this->editing = true;
    }

    public function saveInsurance()
    {
        // 🔒 چک مجدد در action
        $user = Auth::user();
        abort_unless($user?->isAdmin() || $user->projects()->where('project_id', $this->projectId)->exists(), 403);

        $this->validate();

        DB::transaction(function () {
            $data = [
                'project_id' => $this->projectId,
                'company_id' => $this->company_id,
                'start_date' => $this->start_date,
                'end_date' => $this->end_date,
                'execution_duration' => $this->execution_duration,
            ];

            if ($this->editing) {
                $insurance = ProjectInsurance::findOrFail($this->insuranceId);
                $insurance->update($data);
            } else {
                $insurance = ProjectInsurance::create($data);
            }

            $this->handleAttachments($insurance);
        });

        $this->dispatch('saved');
        $this->resetForm();
    }

    public function deleteInsurance(int $id)
    {
        // 🔒 چک مجدد در action
        $user = Auth::user();
        abort_unless($user?->isAdmin() || $user->projects()->where('project_id', $this->projectId)->exists(), 403);

        $insurance = ProjectInsurance::with('attachments')->findOrFail($id);

        foreach ($insurance->attachments as $attachment) {
            Storage::disk('public')->delete($attachment->file_path);
        }

        $insurance->delete();
        $this->dispatch('saved');
    }

    protected function handleAttachments(ProjectInsurance $insurance)
    {
        if (!empty($this->deletedAttachmentIds)) {
            $attachmentsToDelete = InsuranceAttachment::whereIn('id', $this->deletedAttachmentIds)->get();
            foreach ($attachmentsToDelete as $attachment) {
                Storage::disk('public')->delete($attachment->file_path);
                $attachment->delete();
            }
            $this->deletedAttachmentIds = []; // 🔁 پاک کردن بعد از حذف
        }

        if (!empty($this->newAttachments)) {
            $newAttachmentsData = [];
            foreach ($this->newAttachments as $file) {
                $path = $file->store('project_insurances/' . $this->projectId, 'public');
                $newAttachmentsData[] = InsuranceAttachment::make([
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'file_size' => $file->getSize(),
                ]);
            }
            $insurance->attachments()->saveMany($newAttachmentsData);
            $this->newAttachments = []; // 🔁 پاک کردن بعد از ذخیره
        }
    }

    public function removeExistingAttachment(int $attachmentId)
    {
        $this->deletedAttachmentIds[] = $attachmentId;
    }

    public function removeNewAttachment(int $key)
    {
        unset($this->newAttachments[$key]);
        $this->newAttachments = array_values($this->newAttachments);
    }

    public function getPersianDate($date)
    {
        if (!$date) return '—';
        return function_exists('verta')
            ? verta($date)->format('Y/m/d')
            : Jalali\Jalali::parse($date)->format('Y/m/d');
    }

    public function showAttachmentsModal(int $insuranceId)
    {
        // 🔒 امنیت: فقط اعضای پروژه یا ادمین
        $user = Auth::user();
        abort_unless($user?->isAdmin() || $user->projects()->where('project_id', $this->projectId)->exists(), 403);

        $insurance = ProjectInsurance::with('attachments', 'company')
            ->findOrFail($insuranceId);

        $attachments = $insurance->attachments
            ->whereNotIn('id', $this->deletedAttachmentIds)
            ->map(fn($a) => [
                'id' => $a->id,
                'name' => $a->file_name,
                'path' => Storage::disk('public')->url($a->file_path),
                'size' => $a->file_size ? number_format($a->file_size / 1024, 1).' KB' : '—',
            ]);

        $this->dispatch('open-attachments-modal', [
            'attachments' => $attachments->toArray(),
            'title' => 'پیوست‌های بیمه: ' . ($insurance->company->name ?? '—'),
        ]);
    }

    #[Layout('admin.master')]
    public function render()
    {
        $insurances = ProjectInsurance::with('company:id,name', 'attachments')
            ->where('project_id', $this->projectId)
            ->orderByDesc('id')
            ->get();

        return view('livewire.admin.projects.insurance.insurance-detail-create', [
            'insurances' => $insurances,
        ]);
    }
}
