<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Project;
use App\Models\Section;
use Livewire\Attributes\{Url, Computed, Layout};
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\{DB, Session};

class ListProjects extends Component
{
    use WithPagination;

    #[Url(history: true)]
    public string $search = '';

    public ?int $selectedProjectId = null;
    public string $selectedProjectName = '';
    public string $section_name = '';
    public ?string $section_description = null;
    public string $viewMode = 'table';

    protected string $paginationTheme = 'bootstrap';

    public function mount()
    {
        $this->viewMode = Session::get('project_view_mode', 'table');
    }

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function setViewMode(string $mode): void
    {
        $this->viewMode = $mode;
        Session::put('project_view_mode', $mode);
    }

    // --- Computed Properties ---

    #[Computed]
    public function projects()
    {
        return Project::query()
            ->with(['details']) // لود اطلاعات قرارداد
            ->withCount(['sections', 'members', 'purchaseRequests']) // شمارش هوشمند وابستگی‌ها
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('code', 'like', '%' . $this->search . '%');
            })
            ->orderBy('is_active', 'desc') // ابتدا فعال‌ها
            ->orderBy('id', 'desc')
            ->paginate($this->viewMode === 'grid' ? 12 : 10);
    }

    #[Computed]
    public function stats()
    {
        return [
            'total' => Project::count(),
            'active' => Project::where('is_active', true)->count(),
            'total_sections' => Section::count(),
            'total_budget' => Project::all()->sum('budget'), // استفاده از Accessor مدل
        ];
    }

    // --- Actions ---

    public function toggleStatus(int $id)
    {
        $project = Project::findOrFail($id);
        $project->update(['is_active' => !$project->is_active]);
        session()->flash('success', 'وضعیت پروژه تغییر یافت.');
    }

    public function openAddSectionModal(int $projectId, string $projectName): void
    {
        $this->selectedProjectId = $projectId;
        $this->selectedProjectName = $projectName;
        $this->reset(['section_name', 'section_description']);
        $this->dispatch('open-modal');
    }

    public function CreateSection(): void
    {
        $this->validate([
            'section_name' => 'required|string|min:2|max:255',
            'section_description' => 'nullable|string|max:500',
        ]);

        Section::create([
            'name' => $this->section_name,
            'description' => $this->section_description,
            'project_id' => $this->selectedProjectId,
        ]);

        $this->reset(['section_name', 'section_description']);
        $this->dispatch('alert', ['type' => 'success', 'message' => 'بخش جدید با موفقیت اضافه شد.']);
        $this->dispatch('close-modal');
    }

    public function deleteProject(int $projectId): void
    {
        DB::transaction(function () use ($projectId) {
            $project = Project::findOrFail($projectId);
            $project->sections()->delete();
            $project->delete();
        });

        $this->dispatch('alert', ['type' => 'warning', 'message' => 'پروژه و وابستگی‌ها حذف شدند.']);
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.projects.list-projects');
    }
}
