<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Project;
use App\Models\Company;
use App\Models\User;
use Livewire\Component;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class ProjectContractRoles extends Component
{
    use AuthorizesRequests;

    public Project $project;
    public $companies;

    // متغیرهای فیلد Company ID (نقش‌های قراردادی)
    public $main_client_id = '';
    public $sub_client_id = '';
    public $contractor_id = '';
    public $consultant_id = '';

    // متغیرهای فیلد User ID (مسئولین اصلی از آن شرکت)
    public $main_client_user_id = '';
    public $sub_client_user_id = '';
    public $contractor_user_id = '';
    public $consultant_user_id = '';

    // لیست کاربران برای Dropdownهای آبشاری
    public $mainClientUsers = [];
    public $subClientUsers = [];
    public $contractorUsers = [];
    public $consultantUsers = [];

    // 💡 تعریف انواع نقش‌های قراردادی برای دسترسی آسان
    const PARTY_ROLES = ['main_client', 'sub_client', 'contractor', 'consultant'];
    const USER_ROLES = ['main_client_user', 'sub_client_user', 'contractor_user', 'consultant_user'];

    protected $rules = [
        'main_client_id' => 'nullable|exists:companies,id',
        'sub_client_id' => 'nullable|exists:companies,id',
        'contractor_id' => 'nullable|exists:companies,id',
        'consultant_id' => 'nullable|exists:companies,id',

        'main_client_user_id' => 'nullable|exists:users,id',
        'sub_client_user_id' => 'nullable|exists:users,id',
        'contractor_user_id' => 'nullable|exists:users,id',
        'consultant_user_id' => 'nullable|exists:users,id',
    ];

    public function mount(Project $project)
    {
        // 🚨 گارد امنیتی
        // $this->authorize('edit-contract-roles', $project);

        $this->project = $project->load(['details', 'parties', 'managementUsers']);
        $this->companies = Company::select('id', 'name')->orderBy('name')->get();

        // 💡 لود داده‌ها از جداول محوری
        $this->loadInitialData();

        // 💡 فراخوانی آبشاری برای پر کردن Dropdownها پس از لود داده‌ها
        $this->loadUsersForCompany('mainClient', $this->main_client_id);
        $this->loadUsersForCompany('subClient', $this->sub_client_id);
        $this->loadUsersForCompany('contractor', $this->contractor_id);
        $this->loadUsersForCompany('consultant', $this->consultant_id);
    }

    // 💡 متد کمکی برای لود داده‌های موجود در mount
    private function loadInitialData()
    {
        // لود ID شرکت‌ها
        foreach (self::PARTY_ROLES as $role) {
            $party = $this->project->parties->firstWhere('pivot.party_role_type', $role);
            $this->{$role . '_id'} = $party->id ?? null;
        }

        // لود ID کاربران
        foreach (self::USER_ROLES as $role) {
            $user = $this->project->managementUsers->firstWhere('pivot.management_role', $role);
            $this->{$role . '_id'} = $user->id ?? null;
        }
    }


    /**
     * متد عمومی برای واکشی کاربران بر اساس Company ID (آبشاری)
     */
    private function loadUsersForCompany(string $type, $companyId)
    {
        $users = $companyId
            ? User::where('company_id', $companyId)->select('id', 'name')->orderBy('name')->get()
            : collect();

        // تخصیص به پراپرتی لیست کاربران مربوطه
        $this->{$type . 'Users'} = $users;
    }

    // متدهای Updated برای فعال کردن واکشی آبشاری
    public function updatedMainClientId($value) {
        $this->loadUsersForCompany('mainClient', $value);
        $this->main_client_user_id = null;
    }
    public function updatedContractorId($value) {
        $this->loadUsersForCompany('contractor', $value);
        $this->contractor_user_id = null;
    }
    public function updatedConsultantId($value) {
        $this->loadUsersForCompany('consultant', $value);
        $this->consultant_user_id = null;
    }
    public function updatedSubClientId($value) {
        $this->loadUsersForCompany('subClient', $value);
        $this->sub_client_user_id = null;
    }


    public function saveContractRoles()
    {
        $this->validate();

        // 🛑 منطق حذف/ثبت مجدد: اطمینان از اینکه فقط یک نقش اصلی وجود دارد.

        // ۱. حذف نقش‌های قدیمی شرکت
        $this->project->parties()->wherePivotIn('party_role_type', self::PARTY_ROLES)->detach();

        // ۲. حذف نقش‌های قدیمی مسئولین
        $this->project->managementUsers()->wherePivotIn('management_role', self::USER_ROLES)->detach();

        // ۳. ثبت نقش‌های جدید شرکت
        $partiesToAttach = [];
        if ($this->main_client_id) $partiesToAttach[$this->main_client_id] = ['party_role_type' => 'main_client'];
        if ($this->sub_client_id) $partiesToAttach[$this->sub_client_id] = ['party_role_type' => 'sub_client'];
        if ($this->contractor_id) $partiesToAttach[$this->contractor_id] = ['party_role_type' => 'contractor'];
        if ($this->consultant_id) $partiesToAttach[$this->consultant_id] = ['party_role_type' => 'consultant'];

        $this->project->parties()->attach($partiesToAttach);


        // ۴. ثبت نقش‌های جدید مسئولین
        $usersToAttach = [];
        if ($this->main_client_user_id) $usersToAttach[$this->main_client_user_id] = ['management_role' => 'main_client_user'];
        if ($this->sub_client_user_id) $usersToAttach[$this->sub_client_user_id] = ['management_role' => 'sub_client_user'];
        if ($this->contractor_user_id) $usersToAttach[$this->contractor_user_id] = ['management_role' => 'contractor_user'];
        if ($this->consultant_user_id) $usersToAttach[$this->consultant_user_id] = ['management_role' => 'consultant_user'];

        $this->project->managementUsers()->attach($usersToAttach);

        // 💡 بارگذاری مجدد روابط برای به‌روزرسانی UI
        $this->project->load(['parties', 'managementUsers']);
        session()->flash('success', 'نقش‌های قراردادی پروژه با موفقیت به‌روزرسانی شدند.');
    }

    public function render()
    {
        return view('livewire.admin.projects.project-contract-roles')->layout('admin.master');
    }
}
