<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Project;
use App\Models\Section;
use App\Models\PurchaseRequest;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class ProjectDashboard extends Component
{
    use WithPagination;
    protected string $paginationTheme = 'bootstrap';

    #[Url(history: true)]
    public string $search = '';

    public ?int $selectedProjectId = null;
    public string $selectedProjectName = '';
    public string $section_name = '';
    public ?string $section_description = null;

    protected $listeners = ['sectionCreated' => '$refresh'];

    public function updatedSearch(): void { $this->resetPage(); }

    // --- Actions for Modal ---

    public function openAddSectionModal(int $projectId, string $projectName): void {
        $this->selectedProjectId = $projectId;
        $this->selectedProjectName = $projectName;
        $this->reset(['section_name', 'section_description']);
        $this->dispatch('open-modal');
    }

    protected function rules(): array {
        return [
            'section_name' => 'required|string|min:2|max:255',
            'section_description' => 'nullable|string|max:500',
        ];
    }

    public function CreateSection(): void
    {
        $this->validate();

        if (!$this->selectedProjectId) {
            session()->flash('error', 'خطا: پروژه انتخاب نشده است.');
            return;
        }

        Section::create([
            'name' => $this->section_name,
            'description' => $this->section_description,
            'project_id' => $this->selectedProjectId,
        ]);

        $this->reset(['section_name', 'section_description']);
        session()->flash('success', 'بخش جدید با موفقیت اضافه شد.');
        $this->dispatch('close-modal');
        $this->dispatch('sectionCreated');
    }

    public function deleteProject(int $projectId): void
    {
        try {
            DB::transaction(function () use ($projectId) {
                Project::findOrFail($projectId)->delete();
            });
            session()->flash('success', 'پروژه با موفقیت حذف شد.');
            $this->resetPage();
        } catch (\Exception $e) {
            // در محیط واقعی، بهتر است خطا را لاگ کنید.
            session()->flash('error', 'خطا در حذف پروژه. لطفا تمام وابستگی‌ها را بررسی کنید.');
        }
    }

    // --- Computed Properties for Data Fetching ---

    #[Computed]
    public function projects()
    {
        // با حفظ کامل قابلیت شمارش PurchaseRequests
        return Project::query()
            ->withCount(['sections', 'purchaseRequests'])
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('code', 'like', '%' . $this->search . '%')
                    ->orWhereDate('created_at', 'like', '%' . $this->search . '%');
            })
            ->orderBy('id', 'desc')
            ->paginate(8);
    }

    #[Computed]
    public function totalStats(): array
    {
        return [
            'totalSections' => Section::count(),
            'totalPurchaseRequests' => PurchaseRequest::count(),
            'newProjects' => Project::where('created_at', '>=', now()->subDays(7))->count(),
        ];
    }

    #[Computed]
    public function trendData(): array
    {
        // این بخش در ویو فعلی استفاده نشده اما منطق آن حفظ شده است
        $trendLabels = [];
        $projectCounts = [];
        $purchaseCounts = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->startOfDay();
            $label = jdate($date)->format('Y/m/d');

            $projectCounts[] = Project::whereDate('created_at', $date)->count();
            $purchaseCounts[] = PurchaseRequest::whereDate('created_at', $date)->count();

            $trendLabels[] = $label;
        }

        return [
            'labels' => $trendLabels,
            'project' => $projectCounts,
            'purchase' => $purchaseCounts,
        ];
    }

    #[Computed]
    public function latestPurchaseRequests()
    {
        return PurchaseRequest::with('project:id,name,code') // اضافه کردن code به with
        ->latest()
            ->take(5)
            ->get();
    }


    public function render()
    {
        return view('livewire.admin.projects.project-dashboard')->layout('admin.master');
    }
}
