<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Project;
use App\Models\Task;
use App\Models\Section;
use App\Models\User;
use App\Models\PurchaseRequest;
use App\Models\Transmittal;
use App\Models\MasterDocument;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class ProjectDashboardSingle extends Component
{
    public $project;
    public $mainStats;
    public $progressPercent;
    public $tasksStatusData = [], $tasksTrendLabels = [], $tasksTrendData = [];
    public $mdrStatusData = [], $transmittalTrendData = [], $transmittalTrendLabels = [];
    public $purchaseTrendData = [], $purchaseTrendLabels = [];
    public $recentActivities = [];
    public $teamMembers = [];

    // متغیرهای جدید مالی
    public $financialMetrics = [
        'totalBudget' => 0,
        'totalSpent' => 0,
        'totalPettyCash' => 0,
        'remainingBudget' => 0,
        'budgetUtilization' => 0
    ];

    /**
     * متد کمکی برای دریافت تصویر پروفایل کاربر با امنیت بالا
     */
    public function getUserProfilePhotoUrl(mixed $user): string
    {
        $name = is_object($user) && isset($user->name) ? $user->name : 'Unknown';
        $initial = mb_substr($name, 0, 1);
        $encodedInitial = urlencode($initial);
        $default = "https://ui-avatars.com/api/?name={$encodedInitial}&background=EBF4FF&color=2563EB&size=128";

        if (!is_object($user)) return $default;

        if (isset($user->image) && $user->image) {
            if (str_starts_with($user->image, 'http')) {
                return $user->image;
            }
            if (\Storage::disk('public')->exists($user->image)) {
                return \Storage::url($user->image);
            }
        }
        return $default;
    }

    public function mount(Project $project)
    {
        $user = Auth::user();
        if (!$user) abort(403);

        // 🔒 چک دسترسی هوشمند
        if (!$user->isAdmin()) {
            if (!$project->members()->where('users.id', $user->id)->exists()) {
                abort(403, 'شما به این پروژه دسترسی ندارید.');
            }
        }

        // بارگذاری پروژه با روابط کلیدی (شامل آیتم‌های خرید و تنخواه)
        $this->project = $project->load(['sections', 'purchaseRequests.items', 'members', 'pettyCashes']);

        // 💰 ۱. محاسبات مالی (بودجه، هزینه، تنخواه)
        $this->calculateFinancials($project);

        // 📊 ۲. بارگذاری آمار کلی
        $totalTasks    = Task::where('project_id', $project->id)->count();
        $tasksOpen     = Task::where('project_id', $project->id)->where('status', '!=', 'done')->count();
        $tasksDone     = Task::where('project_id', $project->id)->where('status', 'done')->count();
        $sectionsCount = $project->sections->count();
        $reqCount      = $project->purchaseRequests->count();
        $memberCount   = $project->members->count();
        $mdrCount      = MasterDocument::where('project_id', $project->id)->count();
        $transCount    = Transmittal::where('project_id', $project->id)->count();

        $this->mainStats = [
            ['label' => 'کل تسک‌ها',     'value' => $totalTasks,    'icon'=>'ti ti-list-check',     'color'=>'primary'],
            ['label' => 'تسک باز',       'value' => $tasksOpen,     'icon'=>'ti ti-alert-triangle', 'color'=>'warning'],
            ['label' => 'بخش‌ها',        'value' => $sectionsCount, 'icon'=>'ti ti-layers',         'color'=>'info'],
            ['label' => 'درخواست خرید',  'value' => $reqCount,      'icon'=>'ti ti-shopping-cart',  'color'=>'danger'],
            ['label' => 'اعضا',          'value' => $memberCount,   'icon'=>'ti ti-users',          'color'=>'indigo'],
            ['label' => 'مدارک فنی',     'value' => $mdrCount,      'icon'=>'ti ti-file-text',      'color'=>'dark'],
            ['label' => 'بودجه کل (M.R)', 'value' => number_format($this->financialMetrics['totalBudget'] / 1000000, 1), 'icon'=>'ti ti-coin', 'color'=>'success'],
        ];

        // 📈 ۳. پیشرفت فیزیکی و توزیع وضعیت‌ها
        $this->progressPercent = (int) Task::where('project_id', $project->id)->avg('progress_percent') ?: 0;

        $statusMap = ['new'=>'جدید','in_progress'=>'درحال انجام','done'=>'تمام شده','on_hold'=>'متوقف'];
        foreach ($statusMap as $k=>$v) {
            $this->tasksStatusData[] = [
                'label' => $v,
                'value' => Task::where('project_id', $project->id)->where('status', $k)->count()
            ];
        }

        $mdrMap = ['in_progress'=>'درحال کار','approved'=>'تایید شده','pending'=>'انتظار','rejected'=>'اصلاح'];
        foreach ($mdrMap as $k=>$v) {
            $this->mdrStatusData[] = [
                'label' => $v,
                'value' => MasterDocument::where('project_id', $project->id)->where('status', $k)->count()
            ];
        }

        // 🗓️ ۴. داده‌های روند ۳۰ روزه
        $this->loadTrendData($project);

        // 🔥 ۵. فعالیت‌های اخیر (Task, Purchase, MDR, Transmittal)
        $this->loadRecentActivities($project);

        // 👥 ۶. اعضای تیم
        $this->teamMembers = $project->members()->select(['users.id', 'name', 'email', 'image'])->limit(9)->get();
    }

    /**
     * متد اختصاصی برای محاسبات مالی پروژه
     */
    private function calculateFinancials($project)
    {
        // بودجه از Accessor (قرارداد اصلی + الحاقیه‌ها)
        $totalBudget = (float) $project->budget;

        // هزینه‌کرد از درخواست‌های خرید "تایید شده"
        $totalSpent = $project->purchaseRequests()
            ->where('status', 'approved')
            ->withSum('items as total_price', DB::raw('quantity * price'))
            ->get()
            ->sum('total_price');

        // مجموع تنخواه پروژه
        $totalPettyCash = (float) $project->pettyCashes()->sum('amount');

        $currentCost = $totalSpent + $totalPettyCash;
        $remaining = $totalBudget - $currentCost;
        $utilization = $totalBudget > 0 ? round(($currentCost / $totalBudget) * 100, 1) : 0;

        $this->financialMetrics = [
            'totalBudget' => $totalBudget,
            'totalSpent' => $totalSpent,
            'totalPettyCash' => $totalPettyCash,
            'remainingBudget' => $remaining,
            'budgetUtilization' => $utilization
        ];
    }

    private function loadTrendData($project)
    {
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->toDateString();
            $label = verta($date)->format('m/d');
            $this->tasksTrendLabels[] = $label;
            $this->tasksTrendData[] = Task::where('project_id', $project->id)->whereDate('created_at', $date)->count();
            $this->purchaseTrendData[] = PurchaseRequest::where('project_id', $project->id)->whereDate('created_at', $date)->count();
        }
    }

    private function loadRecentActivities($project)
    {
        $recentTasks = Task::where('project_id', $project->id)->latest()->take(3)->get(['title','status','created_at']);
        $recentPurchases = PurchaseRequest::where('project_id', $project->id)->latest()->take(2)->get(['description','status','created_at']);

        $this->recentActivities = [];
        foreach ($recentTasks as $t)
            $this->recentActivities[] = ['type'=>'تسک', 'title'=>$t->title, 'status'=>$t->status, 'color'=>'indigo', 'icon'=>'ti-list-check', 'created_at'=>$t->created_at];
        foreach ($recentPurchases as $p)
            $this->recentActivities[] = ['type'=>'خرید', 'title'=>$p->description, 'status'=>$p->status, 'color'=>'danger', 'icon'=>'ti-shopping-cart', 'created_at'=>$p->created_at];

        usort($this->recentActivities, fn($a, $b) => $b['created_at'] <=> $a['created_at']);
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.projects.project-dashboard-single');
    }
}
