<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Company;
use App\Models\Department;
use App\Models\ProjectType;
use App\Models\ProjectRole;
use App\Models\Project;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class ProjectDetailCreate extends Component
{
    public $project_id;
    public $project_code, $tax_code, $status, $manager_id, $manager_company_id, $manager_department_id, $project_types = [];
    public $roles = [];
    public $roleDuplicateError = false;

    public $companies;
    public $projectTypes;
    public $managerDepartments = [];
    public $managerCandidates = [];

    protected function rules()
    {
        return [
            'project_code'      => 'required|string|max:255',
            'tax_code'          => 'nullable|string|max:255',
            'status'            => 'required|in:شروع نشده,در حال اجرا,اتمام',
            'manager_company_id'=> 'required|exists:companies,id',
            'manager_department_id'=> 'nullable|exists:departments,id',
            'manager_id'        => 'required|exists:users,id',
            'project_types'     => 'required|array|exists:project_types,id',
            'roles.*.role_type' => 'required_with:roles.*.company_id|in:کارفرمای اصلی,کارفرمای فرعی,مشاور',
            'roles.*.company_id'=> 'required_with:roles.*.role_type|exists:companies,id',
        ];
    }

    public function mount($projectId)
    {
        // 🔒 امنیت: فقط ادمین یا اعضای پروژه (user_roles) دسترسی دارند
        $user = Auth::user();
        if (!$user) {
            abort(403);
        }

        $project = Project::findOrFail($projectId);
        $this->project_id = $projectId;

        if (!$user->isAdmin() && !$user->projects()->where('project_id', $project->id)->exists()) {
            // 💡 اختیاری: فعال‌سازی دسترسی برای شرکت‌های ذینفع (کارفرما/پیمانکار/مشاور)
            // if (!$user->company || !$project->parties()->where('company_id', $user->company->id)->exists()) {
            abort(403, 'شما به این پروژه دسترسی ندارید.');
            // }
        }

        // ✅ فقط در صورت گذر از چک بالا، اطلاعات بارگذاری می‌شوند:
        $this->companies = Company::all();
        $this->projectTypes = ProjectType::all();

        $detail = $project->details;
        $this->project_code = $project->code;
        $this->tax_code = $detail?->tax_code;
        $this->status   = $detail?->status;
        $this->manager_company_id = $detail?->main_client_id;
        $this->manager_department_id = $detail?->manager_department_id;
        $this->manager_id = $detail?->manager_id;
        $this->project_types = $project->projectTypes->pluck('id')->toArray();
        $this->roles = $project->roles->map(fn($role) => [
            'role_type'      => $role->role_type,
            'company_id'     => $role->company_id,
        ])->toArray();

        $this->updateManagerDependencies();
    }

    public function render()
    {
        return view('livewire.admin.projects.project-detail-create')->layout('admin.master');
    }

    public function updateManagerDependencies()
    {
        $this->managerDepartments = Department::where('company_id', $this->manager_company_id)
            ->where('status', 'active')->get();

        $this->managerCandidates = User::where('company_id', $this->manager_company_id)
            ->when($this->manager_department_id, function($q) {
                $q->where('department_id', $this->manager_department_id);
            })
            ->where('is_active', true)
            ->get();

        if (
            $this->manager_id &&
            !$this->managerCandidates->pluck('id')->contains($this->manager_id)
        ) {
            $this->manager_id = null;
        }
    }

    public function updatedManagerCompanyId()
    {
        $this->manager_department_id = null;
        $this->manager_id = null;
        $this->updateManagerDependencies();
    }

    public function updatedManagerDepartmentId()
    {
        $this->manager_id = null;
        $this->updateManagerDependencies();
    }

    public function addRole()
    {
        $this->roles[] = ['role_type' => '', 'company_id' => ''];
    }

    public function removeRole($index)
    {
        unset($this->roles[$index]);
        $this->roles = array_values($this->roles);
        $this->roleDuplicateError = false;
    }

    protected function hasDuplicateRole()
    {
        $filtered = array_filter($this->roles, fn($r) => $r['role_type'] && $r['company_id']);
        $unique = array_map(fn($r) => $r['role_type'].'_'.$r['company_id'], $filtered);
        return count($unique) !== count(array_unique($unique));
    }

    public function save()
    {
        // 🔐 چک مجدد در action (برای جلوگیری از حمله مستقیم)
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project_id)->exists())) {
            abort(403);
        }

        $this->validate();

        if ($this->hasDuplicateRole()) {
            $this->roleDuplicateError = true;
            session()->flash('error', 'نقش یا شرکت تکراری است.');
            return;
        }

        $project = Project::find($this->project_id);
        $project->update(['code' => $this->project_code]);

        $project->details()->updateOrCreate(
            [],
            [
                'project_code'          => $this->project_code,
                'tax_code'              => $this->tax_code,
                'status'                => $this->status,
                'manager_id'            => $this->manager_id,
                'main_client_id'        => $this->manager_company_id,
                'manager_department_id' => $this->manager_department_id,
            ]
        );

        $project->projectTypes()->sync($this->project_types);

        $project->roles()->delete();
        foreach ($this->roles as $role) {
            if ($role['role_type'] && $role['company_id']) {
                ProjectRole::create([
                    'project_id'    => $project->id,
                    'role_type'     => $role['role_type'],
                    'company_id'    => $role['company_id'],
                ]);
            }
        }

        session()->flash('message', 'جزئیات پروژه با موفقیت ذخیره شد.');
        $this->resetForm();
    }

    private function resetForm()
    {
        $this->resetErrorBag();
        $this->resetValidation();
        $this->project_code = null;
        $this->tax_code = null;
        $this->status = null;
        $this->manager_company_id = null;
        $this->manager_id = null;
        $this->manager_department_id = null;
        $this->project_types = [];
        $this->roles = [];
        $this->managerDepartments = [];
        $this->managerCandidates = [];
        $this->roleDuplicateError = false;
    }
}
