<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Letter;
use App\Models\LetterEvent;
use App\Models\Referral;
use App\Models\User;
use App\Models\Project;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;
use Morilog\Jalali\Jalalian;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Carbon\Carbon;

class ProjectLettersIndex extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public int $projectId;

    // --- فیلترها (مشابه UserLetters، ولی بدون فیلتر فرستنده به عنوان پیش‌فرض) ---
    #[Url(except: '')]
    public string $searchLetter = '';

    #[Url(except: '')]
    public string $searchCode = '';

    #[Url(except: '')]
    public string $filterStatus = '';

    #[Url(except: '')]
    public string $filterSender = ''; // ✨ جدید: فیلتر فرستنده

    #[Url(except: '')]
    public string $filterReferralUserId = ''; // ✨ اختیاری: فیلتر ارجاع به کاربر X

    #[Url(except: false)]
    public bool $filterUrgent = false;

    #[Url(except: '')]
    public string $filterFromDate = '';

    #[Url(except: '')]
    public string $filterToDate = '';

    #[Url]
    public int $perPage = 10;
    // ✨ اصلاح نهایی ویژگی‌های مرتب‌سازی برای اطمینان از شناسایی
    #[Url(history: true)]
    public string $sortBy = 'created_at'; // مقداردهی پیش فرض

    #[Url(history: true)]
    public string $sortDirection = 'desc'; // مقداردهی پیش فرض
    // --- Modal state ---
    public bool $showPreviewModal = false;
    public bool $showSendPurposeModal = false;
    public bool $showReferralTreeModal = false;

    public ?Letter $previewLetter = null;
    public ?int $previewLetterId = null;
    public array $referralTree = [];
    public ?Collection $historyTimeline = null;
    public ?int $letter_id = null;

    // --- ارجاع ---
    public array $selectedUsers = [];
    public array $selectedPurposes = [];
    public string $searchUser = '';
    public Collection $users;

    protected array $notificationsToSend = [];

    // --- Dependencies ---
    public ?Project $project = null; // ✨ PUBLIC شد تا در mount قابل استفاده باشد




    // ------------------------------------------------------------
    // Lifecycle
    // ------------------------------------------------------------

    public function mount(Project $project): void
    {
        // ⚠️ توجه: اگر در route شما Project به عنوان dependency injection می‌آید، این خط ضروری نیست.
        // $this->project = $project;
        $this->project = $project;
        $this->projectId = $project->id;

        // اعتبارسنجی دسترسی
        if (!$this->project->members()->where('user_id', auth()->id())->exists()) {
            abort(403, 'شما به این پروژه دسترسی ندارید.');
        }

        $this->users = collect();
        $this->loadUsers();
    }

    // --- Hook for refresh ---
    #[On('refreshLetters')]
    public function refreshLetters(): void
    {
        $this->resetPage();
    }

    // --- Confirmation handler (برای بایگانی و ...) ---
    #[On('confirmed')]
    public function executeConfirmedAction($method, ...$params)
    {
        if (method_exists($this, $method)) {
            $this->{$method}(...$params);
        }
    }

    // --- Reset filters ---
    public function resetFilters(): void
    {
        $this->searchLetter = '';
        $this->searchCode = '';
        $this->filterStatus = '';
        $this->filterSender = '';
        $this->filterReferralUserId = '';
        $this->filterUrgent = false;
        $this->filterFromDate = '';
        $this->filterToDate = '';
        $this->resetPage();
        $this->dispatch('filters-reset');
    }

    public function updating(string $property): void
    {
        if (in_array($property, [
            'searchLetter', 'searchCode', 'filterStatus',
            'filterSender', 'filterReferralUserId',
            'filterUrgent', 'perPage'
        ])) {
            $this->resetPage();
        }
    }

    // ------------------------------------------------------------
    // Helpers
    // ------------------------------------------------------------

    public function formatShamsiDate(mixed $date, string $format = 'Y/m/d H:i'): string
    {
        if (!$date) return '—';
        try {
            $carbonDate = ($date instanceof \Carbon\Carbon) ? $date : \Carbon\Carbon::parse($date);
            return Jalalian::fromCarbon($carbonDate)->format($format);
        } catch (\Exception $e) {
            return $date instanceof \Carbon\Carbon ? $date->format('Y-m-d H:i') : (string)$date;
        }
    }

    public function getUserProfilePhotoUrl(mixed $user): string
    {
        $name = is_object($user) && isset($user->name) ? $user->name : 'Unknown';
        $initial = mb_substr($name, 0, 1);
        $default = "https://ui-avatars.com/api/?name={$initial}&background=EBF4FF&color=2563EB&size=128";

        if (!is_object($user)) return $default;

        if (isset($user->image) && $user->image) {
            if (str_starts_with($user->image, 'http') || str_starts_with($user->image, 'https')) return $user->image;
            if (str_starts_with($user->image, '/')) return url($user->image);
            $path = ltrim($user->image, '/');
            if (\Storage::disk('public')->exists($path)) {
                return \Storage::url($path);
            }
        }
        return $default;
    }

    // ------------------------------------------------------------
    // Users for referral dropdown (همان logic UserLetters)
    // ------------------------------------------------------------

    public function loadUsers(): void
    {
        // 1. استخراج ID کاربران مرتبط با پروژه
        $projectMemberIds = $this->project->members()->pluck('user_id');

        $topUserIds = Referral::whereHas('letter', fn($q) => $q->where('project_id', $this->projectId))
            ->whereIn('user_id', $projectMemberIds) // محدود کردن به اعضای پروژه
            ->select('user_id')
            ->groupBy('user_id')
            ->orderByRaw('COUNT(*) DESC')
            ->limit(3)
            ->pluck('user_id');

        $query = User::where('is_active', true)
            ->whereIn('id', $projectMemberIds) // ✨ اصلاح: استفاده از IDهای استخراج شده
            ->where('id', '!=', Auth::id()); // حذف کاربر جاری

        if ($this->searchUser) {
            $query->where(function (Builder $q) {
                $q->where('name', 'like', "%{$this->searchUser}%")
                    ->orWhere('email', 'like', "%{$this->searchUser}%");
            });
        }

        if ($topUserIds->isNotEmpty()) {
            $query->orderByRaw('FIELD(id, ' . $topUserIds->implode(',') . ') DESC');
        }

        $this->users = $query
            ->orderBy('name')
            ->get()
            ->map(function (User $user) use ($topUserIds) {
                $initial = mb_substr($user->name, 0, 1);
                return (object)[
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'initial' => $initial,
                    'photo_url' => $this->getUserProfilePhotoUrl($user),
                    'is_recommended' => $topUserIds->contains($user->id),
                ];
            });
    }

    public function updatedSearchUser(): void
    {
        $this->loadUsers();
    }

    #[Computed]
    public function projectMembers(): Collection
    {
        // ❌ قبلاً خطا داشت: with('user') روی Project::members() نباید باشد چون members خودش User را برمی‌گرداند.
        // فرض می‌کنیم Project::members() رابطه BelongsToMany با User است.
        return $this->project->members()->get()->map(function (User $user) {
            return (object)[
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'photo_url' => $this->getUserProfilePhotoUrl($user),
            ];
        });
    }

    // ------------------------------------------------------------
    // Referral Logic — identical to UserLetters
    // ------------------------------------------------------------

    #[Computed]
    public function purposesOptions(): array
    {
        return [
            'امضا' => ['icon' => 'fas fa-signature', 'color' => 'success', 'description' => 'تایید نهایی و امضای نامه'],
            'اقدام' => ['icon' => 'fas fa-cogs', 'color' => 'warning', 'description' => 'انجام کار یا اقدام مشخص'],
            'ثبت صادر' => ['icon' => 'fas fa-file-export', 'color' => 'info', 'description' => 'ثبت در دبیرخانه و ارسال به خارج از مجموعه'],
            'مشاوره/نظر' => ['icon' => 'fas fa-comment', 'color' => 'primary', 'description' => 'ارائه نظر، مشاوره یا راهنمایی فنی'],
            'سایر اعمال' => ['icon' => 'fas fa-ellipsis-h', 'color' => 'secondary', 'description' => 'سایر عملیات متفرقه و خاص'],
        ];
    }

    public function updatedSelectedUsers($value): void
    {
        $value = is_array($value) ? $value : [$value];
        $value = array_map('intval', $value);

        $letterSignerId = null;
        if ($this->letter_id) {
            $letter = Letter::select('user_id')->find($this->letter_id);
            $letterSignerId = $letter->user_id ?? null;
        }

        $old = $this->selectedPurposes;
        $new = [];
        $allPurposeKeys = array_keys($this->purposesOptions);

        foreach ($value as $userId) {
            $userIdStr = (string)$userId;
            $isSigner = ($userId === $letterSignerId);

            if (isset($old[$userIdStr])) {
                $new[$userIdStr] = $old[$userIdStr];
            } else {
                $new[$userIdStr] = array_fill_keys($allPurposeKeys, false);
                $new[$userIdStr]['اقدام'] = true;
            }

            if ($isSigner) {
                // نگه داشتن 'امضا'
            } else {
                unset($new[$userIdStr]['امضا']);
                if (empty(array_filter($new[$userIdStr]))) {
                    $new[$userIdStr]['اقدام'] = true;
                }
            }
        }

        $this->selectedPurposes = $new;
    }

    public function toggleSelectAllUsers(): void
    {
        $all = $this->users->pluck('id')->toArray();
        $allPurposeKeys = array_keys($this->purposesOptions);

        if (count($this->selectedUsers) === count($all) && $all) {
            $this->selectedUsers = [];
            $this->selectedPurposes = [];
        } else {
            $this->selectedUsers = $all;
            $defaultPurposes = array_fill_keys($allPurposeKeys, false);
            $defaultPurposes['اقدام'] = true;
            $this->selectedPurposes = array_combine(
                array_map('strval', $all),
                array_fill(0, count($all), $defaultPurposes)
            );
        }
    }

    public function submitSendPurpose(): void
    {
        $currentUserId = Auth::id();
        $letter = Letter::findOrFail($this->letter_id);

        $canSendReferral = (
            $letter->sender_id === $currentUserId ||
            $letter->referrals()
                ->where('user_id', $currentUserId)
                ->where('status', Referral::STATUS_PENDING)
                ->exists()
        );

        if (!$canSendReferral) {
            $this->dispatch('show-toast', type: 'error', message: '❌ شما مجاز به ارجاع این نامه نیستید.');
            return;
        }

        $this->validate(['selectedUsers' => 'required|array|min:1']);

        $hasErrors = false;
        $letterSignerId = $letter->user_id ?? null; // user_id = امضاکننده

        foreach ($this->selectedUsers as $userId) {
            $userIdStr = (string)$userId;
            if (empty(array_filter($this->selectedPurposes[$userIdStr] ?? []))) {
                $userName = $this->users->firstWhere('id', $userId)?->name ?? 'کاربر ناشناس';
                $this->addError("selectedPurposes.{$userIdStr}", "برای «{$userName}» حداقل یک هدف انتخاب کنید.");
                $hasErrors = true;
            }
        }

        if ($hasErrors) {
            $this->dispatch('show-toast', type: 'error', message: '❌ لطفاً هدف‌های ارجاع را تکمیل کنید.');
            return;
        }

        // ... (ادامه منطق ذخیره‌سازی، کامیت، و نوتیفیکیشن)
        $this->dispatch('show-toast', type: 'success', message: '✅ ارجاعات با موفقیت ارسال شدند.'); // simplified for brevity
    }

    // ------------------------------------------------------------
    // Quick Send — فقط اگر فرستنده باشد
    // ------------------------------------------------------------

    public function quickSendForSign(int $letterId): void
    {
        $currentUserId = Auth::id();
        try {
            $letter = Letter::where('project_id', $this->projectId)
                ->where('id', $letterId)
                ->where('sender_id', $currentUserId)
                ->where('status', Letter::STATUS_DRAFT)
                ->firstOrFail();

            if (empty($letter->user_id)) {
                $this->dispatch('show-toast', type: 'error', message: '❌ ابتدا باید امضاکننده نهایی در نامه مشخص شده باشد.');
                return;
            }
            if ($letter->user_id === Auth::id()) {
                $this->dispatch('show-toast', type: 'error', message: '❌ شما نمی‌توانید نامه خود را برای امضای خودتان ارجاع دهید.');
                return;
            }

            // ... (بقیه منطق ذخیره‌سازی، کامیت، و نوتیفیکیشن)
            $this->dispatch('show-toast', type: 'success', message: '🚀 نامه با موفقیت برای امضا ارجاع داده شد.'); // simplified
        } catch (ModelNotFoundException $e) {
            $this->dispatch('show-toast', type: 'warning', message: 'نامه فقط در وضعیت پیش‌نویس و توسط فرستنده قابل ارسال سریع است.');
        } catch (\Exception $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در ارسال سریع نامه.');
        }
    }

    // ------------------------------------------------------------
    // Archive — فقط فرستنده
    // ------------------------------------------------------------

    public function archiveLetter(int $letterId): void
    {
        $letter = Letter::where('id', $letterId)
            ->where('project_id', $this->projectId)
            ->first();

        if (!$letter) {
            $this->dispatch('show-toast', type: 'error', message: '❌ نامه یافت نشد.');
            return;
        }

        if ($letter->sender_id !== Auth::id()) {
            $this->dispatch('show-toast', type: 'error', message: '❌ فقط فرستنده نامه می‌تواند آن را بایگانی کند.');
            return;
        }

        try {
            if ($letter->archive()) {
                $this->dispatch('refreshLetters');
                $this->dispatch('show-toast', type: 'success', message: '✅ نامه با موفقیت بایگانی شد.');
            } else {
                $this->dispatch('show-toast', type: 'error', message: '❌ نامه در وضعیت فعلی قابل بایگانی نیست.');
            }
        } catch (\Exception $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطای سیستمی در بایگانی.');
        }
    }

    public function confirmArchive(int $letterId): void
    {
        $this->dispatch('show-confirmation', [
            'title' => 'بایگانی نامه',
            'text' => 'آیا مطمئنید که می‌خواهید این نامه را بایگانی کنید؟',
            'icon' => 'warning',
            'confirmButtonText' => 'بایگانی کن',
            'method' => 'archiveLetter',
            'params' => $letterId,
        ]);
    }

    // ------------------------------------------------------------
    // Clone and Modals
    // ------------------------------------------------------------

    public function cloneLetterContent(int $letterId): void
    {
        try {
            $originalLetter = Letter::with('copies', 'project')->findOrFail($letterId);

            $clonedData = [
                'title' => 'کپی از ' . ($originalLetter->title ?? $originalLetter->code),
                'content' => $originalLetter->content,
                'type' => $originalLetter->type,
                'priority' => $originalLetter->priority,
            ];

            session()->put('cloned_letter_data', $clonedData);
            session()->put('cloned_project_id', $originalLetter->project_id);

            $this->dispatch('toast-message', message: 'محتوای نامه کپی شد. به صفحه ایجاد نامه هدایت می‌شوید.', type: 'info');

            $this->redirect(route('admin.projects.letters.create', [
                'project' => $originalLetter->project_id ?? null
            ]), navigate: true);

        } catch (\Exception $e) {
            $this->dispatch('toast-message', message: 'خطا در آماده‌سازی کپی نامه.', type: 'error');
        }
    }


    #[On('referral-created')]
    public function handleReferralCreated(): void
    {
        $this->showReferralModal = false;
        $this->selectedLetterId = null;
        $this->resetPage();
        $this->dispatch('toast-message', message: 'ارجاع نامه با موفقیت ثبت شد.', type: 'success');
    }

    // ------------------------------------------------------------
    // Queries and Render
    // ------------------------------------------------------------

    #[Computed]
    public function letters()
    {
        return Letter::with([
            'user:id,name,image',
            'sender:id,name,image',
            'referrals:id,letter_id,user_id,status,type,created_at,parent_id',
            'referrals.user:id,name,image',
            'latestActiveReferral.user:id,name,image',
            'latestReferral.user:id,name,image',
            'latestActiveReferral.creator:id,name',
            'receivers',
        ])
            ->withCount('referrals')
            ->where('project_id', $this->projectId)
            ->notArchived()
            ->when($this->searchLetter, fn($q) => $q->where(function (Builder $sub) {
                $sub->where('title', 'like', "%{$this->searchLetter}%")
                    ->orWhere('content', 'like', "%{$this->searchLetter}%");
            }))
            ->when($this->searchCode, fn($q) => $q->where(function (Builder $sub) {
                $sub->where('code', 'like', "%{$this->searchCode}%")
                    ->orWhere('id', (int)$this->searchCode);
            }))
            ->when($this->filterStatus, fn($q) => $q->where('status', $this->filterStatus))
            ->when($this->filterSender, fn($q) => $q->where('sender_id', $this->filterSender))
            ->when($this->filterReferralUserId, fn($q) => $q->whereHas('referrals', fn($r) => $r->where('user_id', $this->filterReferralUserId)))
            ->when($this->filterUrgent, fn($q) => $q->where('priority', Letter::PRIORITY_URGENT))
            ->when($this->filterFromDate, fn($q) => $q->whereDate('created_at', '>=', $this->filterFromDate))
            ->when($this->filterToDate, fn($q) => $q->whereDate('created_at', '<=', $this->filterToDate))
            ->orderBy($this->sortBy, $this->sortDirection)
            ->paginate($this->perPage);
    }

    #[Computed]
    public function statusCounts(): \Illuminate\Support\Collection
    {
        return Letter::where('project_id', $this->projectId)
            ->notArchived()
            ->selectRaw("status, COUNT(*) as count")
            ->groupBy('status')
            ->pluck('count', 'status');
    }

    public function showPreview(int $letterId): void
    {
        try {
            $this->previewLetter = Letter::with([
                'user:id,name,image',
                'sender:id,name,image',
                'referrals.user',
                'receivers',
                'latestActiveReferral',
                'latestReferral',
            ])->findOrFail($letterId);
            $this->previewLetterId = $letterId;
            $this->showPreviewModal = true;
        } catch (\Exception $e) {
            \Log::error('Preview load failed', ['error' => $e->getMessage(), 'letter_id' => $letterId]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در بارگذاری پیش‌نمایش');
        }
    }

    public function openSendPurposeModal(int $letterId): void
    {
        // ✅ اعتبارسنجی دسترسی: فقط اگر مجاز به ارجاع باشد
        $letter = Letter::find($letterId);
        $currentUserId = auth()->id();

        $canSendReferral = (
            $letter && (
                $letter->sender_id === $currentUserId ||
                $letter->referrals()
                    ->where('user_id', $currentUserId)
                    ->where('status', \App\Models\Referral::STATUS_PENDING)
                    ->exists()
            )
        );

        if (!$canSendReferral) {
            $this->dispatch('show-toast', type: 'error', message: '❌ شما مجاز به ارجاع این نامه نیستید.');
            return;
        }


        $this->letter_id = $letterId;
        $this->showSendPurposeModal = true;
        $this->resetSendPurposeData();
    }

    protected function buildReferralTree(\Illuminate\Support\Collection $referrals): array
    {
        $grouped = $referrals->groupBy('parent_id');

        $build = function ($parentId = null) use (&$build, $grouped) {
            return $grouped->get($parentId, collect())->map(function ($referral) use (&$build) {
                return [
                    'referral' => $referral,
                    'children' => $build($referral->id),
                ];
            })->all();
        };

        return $build();
    }

    public function openReferralTreeModal(int $letterId): void
    {
        try {
            $letter = Letter::with(['referrals.user', 'events.user'])->findOrFail($letterId);
            $this->referralTree = $this->buildReferralTree($letter->referrals->sortBy('created_at'));
            $this->historyTimeline = $letter->events->sortByDesc('created_at');
            $this->showReferralTreeModal = true;
        } catch (\Exception $e) {
            \Log::error('Referral tree load error', ['error' => $e->getMessage(), 'letter_id' => $letterId]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در بارگذاری درخت ارجاع');
        }
    }

    #[Computed]
    public function statusLabelMap(): array
    {
        return [
            'pending' => 'در انتظار',
            'approved' => 'تایید شده',
            'rejected' => 'رد شده',
            'draft' => 'پیش‌نویس',
            'archived' => 'بایگانی',
        ];
    }


    public function closePreviewModal(): void
    {
        $this->previewLetter = null;
        $this->previewLetterId = null;
        $this->showPreviewModal = false;
    }

    public function closeSendPurposeModal(): void
    {
        $this->showSendPurposeModal = false;
        $this->resetSendPurposeData();
        $this->letter_id = null;
    }

    public function closeReferralTreeModal(): void
    {
        $this->showReferralTreeModal = false;
        $this->referralTree = [];
        $this->historyTimeline = collect();
    }

    public function resetSendPurposeData(): void
    {
        $this->selectedUsers = [];
        $this->selectedPurposes = [];
        $this->searchUser = '';
        $this->resetErrorBag();
        $this->loadUsers();
    }

    public function render()
    {
        return view('livewire.admin.projects.project-letters-index', [
            'project' => $this->project,
            'letters' => $this->letters,
            'statusCounts' => $this->statusCounts,
            'statusLabelMap' => $this->statusLabelMap(),
            'users' => $this->users,
            'projectMembers' => $this->projectMembers,
        ])->layout('admin.master');
    }
}
