<?php

namespace App\Livewire\Admin\Projects;

use App\Models\Project;
use App\Models\Section;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

class ProjectSection extends Component
{
    use WithPagination;

    public Project $project;

    #[Url(as: 'q', history: true)]
    public string $search = '';

    // متغیرهای فرم
    public string $new_section_name = '';
    public ?string $new_section_description = null;
    public ?int $edit_section_id = null;
    public string $edit_section_name = '';
    public ?string $edit_section_description = null;

    #[Url(as: 'view', history: true)]
    public string $viewType = 'card';

    #[Url(as: 'status', history: true)]
    public string $filterStatus = '';

    protected string $paginationTheme = 'bootstrap';

    protected function rules(): array
    {
        return [
            'new_section_name' => 'required|min:2|max:255',
            'new_section_description' => 'nullable|max:500',
            'edit_section_name' => 'required|min:2|max:255',
            'edit_section_description' => 'nullable|max:500',
        ];
    }

    public function mount(Project $project): void
    {
        $this->project = $project;

        // 🔒 امنیت: فقط ادمین یا اعضای پروژه (از طریق user_roles) مجازند
        $user = Auth::user();

        if (!$user) {
            abort(403);
        }

        if (!$user->isAdmin() &&
            !$user->projects()->where('project_id', $project->id)->exists()) {
            // 💡 اختیاری: اگر کارمندان شرکت‌های ذینفع (کارفرما/پیمانکار) هم باید دسترسی داشته باشند:
            // if (!$user->company || !$project->parties()->where('company_id', $user->company->id)->exists()) {
            abort(403, 'شما به این پروژه دسترسی ندارید.');
            // }
        }
    }

    public function resetForms(): void
    {
        $this->reset([
            'new_section_name',
            'new_section_description',
            'edit_section_id',
            'edit_section_name',
            'edit_section_description',
        ]);
        $this->resetValidation();
    }
    #[Computed]
    public function stats()
    {
        // روش صحیح برای شمارش کل فایل‌های مرتبط با پروژه‌ از طریق بخش‌ها
        $totalFiles = \App\Models\ProjectFile::whereIn(
            'section_id',
            $this->project->sections()->pluck('id')
        )->count();

        return [
            'total' => $this->project->sections()->count(),
            'active' => $this->project->sections()->where('status', 'active')->count(),
            'with_files' => $this->project->sections()->has('files')->count(),
            'total_files' => $totalFiles, // این مقدار را جایگزین sum قبلی کنید
        ];
    }

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function updatedFilterStatus(): void
    {
        $this->resetPage();
    }

    public function updatedViewType(): void
    {
        $this->resetPage();
    }

    // افزودن بخش جدید — ✅ امنیت در mount کافی است، ولی برای اطمینان بیشتر:
    public function createSection(): void
    {
        // ❗ اینجا دوباره چک کنیم (اگر کاربر session را عوض کند یا دستی POST کند)
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project->id)->exists())) {
            abort(403);
        }

        $this->validate([
            'new_section_name' => $this->rules()['new_section_name'],
            'new_section_description' => $this->rules()['new_section_description'],
        ]);

        try {
            $this->project->sections()->create([
                'name' => $this->new_section_name,
                'description' => $this->new_section_description,
                'status' => 'planning',
            ]);

            $this->resetForms();
            session()->flash('success', 'بخش جدید با موفقیت اضافه شد.');
            $this->dispatch('close-add-section-modal');
            $this->resetPage();
        } catch (\Exception $e) {
            session()->flash('error', 'خطا در ثبت بخش: ' . $e->getMessage());
        }
    }

    public function editSection(int $sectionId): void
    {
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project->id)->exists())) {
            abort(403);
        }

        $section = Section::findOrFail($sectionId);
        $this->edit_section_id = $section->id;
        $this->edit_section_name = $section->name;
        $this->edit_section_description = $section->description;
        $this->dispatch('open-edit-section-modal');
    }

    public function updateSection(): void
    {
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project->id)->exists())) {
            abort(403);
        }

        $this->validate([
            'edit_section_name' => $this->rules()['edit_section_name'],
            'edit_section_description' => $this->rules()['edit_section_description'],
        ]);

        try {
            $section = Section::findOrFail($this->edit_section_id);
            $section->update([
                'name' => $this->edit_section_name,
                'description' => $this->edit_section_description,
            ]);
            session()->flash('success', 'بخش با موفقیت ویرایش شد.');
        } catch (\Exception $e) {
            session()->flash('error', 'خطا در به‌روزرسانی: ' . $e->getMessage());
        }

        $this->resetForms();
        $this->dispatch('close-edit-section-modal');
    }

    public function deleteSection(int $sectionId): void
    {
        $user = Auth::user();
        if (!$user || (!$user->isAdmin() && !$user->projects()->where('project_id', $this->project->id)->exists())) {
            abort(403);
        }

        try {
            Section::findOrFail($sectionId)->delete();
            session()->flash('success', 'بخش با موفقیت حذف شد.');
        } catch (\Exception $e) {
            session()->flash('error', 'خطا در حذف بخش.');
        }
        $this->resetPage();
    }

    #[Layout('admin.master')]
    public function render()
    {
        $sectionsQuery = $this->project->sections();

        $sectionsQuery->when($this->search, function ($q) {
            $q->where('name', 'like', "%{$this->search}%")
                ->orWhere('description', 'like', "%{$this->search}%");
        });

        $sectionsQuery->when($this->filterStatus, function ($q) {
            $q->where('status', $this->filterStatus);
        });

        $sectionsQuery->orderBy('created_at', 'desc');
        $sections = $sectionsQuery->paginate(9);

        return view('livewire.admin.projects.project-section', [
            'sections' => $sections,
        ]);
    }
}
