<?php

namespace App\Livewire\Admin\Projects;

use Livewire\Component;
use App\Models\ProjectWallPost;
use App\Models\Project;
use App\Models\WallPostAttachment;
use App\Models\WallPostComment;
use Livewire\Attributes\Rule;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Carbon\Carbon;

class ProjectWall extends Component
{
    use WithPagination;
    use WithFileUploads;

    public Project $project;

    // State for Edit Mode
    public $editingPostId = null;
    public string $editingContent = '';
    public string $editingPostType = 'status';
    public $editingAttachments = []; // New attachments during edit
    public $deletedAttachmentIds = []; // IDs of attachments to delete

    // For registering comment content (keyed by post ID)
    public $newCommentText = [];

    // Modal Comments State
    public $showAllCommentsModal = false;
    public $currentPostComments = [];
    public $currentPostIdForComments = null;

    // Property for new post
    #[Rule('required|string|min:5|max:1000', message: 'محتوا الزامی است و حداقل ۵ کاراکتر نیاز دارد.')]
    public string $newPostContent = '';

    #[Rule('required|in:status,report,milestone,issue,visit,knowledge')]
    public string $newPostType = 'status';

    #[Rule([
        'attachments' => 'nullable|array|max:5',
        'attachments.*' => 'file|mimes:pdf,jpg,jpeg,png,zip,rar,doc,docx,xls,xlsx|max:5120',
    ], message: [
        'attachments.max' => 'شما مجاز به آپلود حداکثر 5 فایل هستید.',
        'attachments.*.max' => 'حجم هر فایل نباید بیشتر از 5 مگابایت باشد.',
    ])]
    public $attachments = [];

    public string $filterType = 'all';
    public string $searchQuery = '';

    // Available types
    public array $availableTypes = [
        'status' => 'گزارش وضعیت',
        'report' => 'گزارش رسمی',
        'milestone' => 'نقطه عطف',
        'issue' => 'مشکل/مانع',
        'visit' => 'بازدید',
        'knowledge' => 'انتقال دانش',
    ];

    public function mount(Project $project)
    {
        $this->project = $project;
    }

    public function updatedFilterType() { $this->resetPage(); }
    public function updatedSearchQuery() { $this->resetPage(); }

    // Post a comment
    public function postComment(int $postId)
    {
        $commentContent = $this->newCommentText[$postId] ?? null;
        if (!Auth::check() || empty($commentContent) || strlen($commentContent) < 2) {
            session()->flash('wall-error', 'محتوای نظر نمی‌تواند خالی باشد.');
            return;
        }
        WallPostComment::create([
            'project_wall_post_id' => $postId,
            'user_id' => Auth::id(),
            'content' => $commentContent,
        ]);
        $this->newCommentText[$postId] = '';
        // No resetPage() to avoid full refresh; rely on polling or real-time
    }

    // Post new update
    public function postUpdate()
    {
        if (!Auth::check()) {
            session()->flash('wall-error', 'برای ثبت وضعیت ابتدا وارد شوید.');
            return;
        }
        $this->validate();
        $post = ProjectWallPost::create([
            'project_id' => $this->project->id,
            'user_id' => Auth::id(),
            'content' => $this->newPostContent,
            'type' => $this->newPostType,
        ]);
        if (!empty($this->attachments)) {
            foreach ($this->attachments as $file) {
                $path = $file->store('project_wall/' . $this->project->id, 'public');
                $post->attachments()->create([
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'file_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }
        $this->newPostContent = '';
        $this->attachments = [];
        $this->newPostType = 'status';
        session()->flash('wall-success', 'وضعیت جدید با موفقیت ثبت شد.');
        $this->resetPage();
    }

    public function startEdit(ProjectWallPost $post)
    {
        if (!$this->canEdit($post)) {
            session()->flash('wall-error', 'مهلت ویرایش این پست (۱۰ روز) به پایان رسیده است.');
            return;
        }
        $this->editingPostId = $post->id;
        $this->editingContent = $post->content;
        $this->editingPostType = $post->type;
        $this->editingAttachments = [];
        $this->deletedAttachmentIds = [];
    }

    public function updatePost()
    {
        if (!Auth::check() || !$this->editingPostId) return;
        $post = ProjectWallPost::find($this->editingPostId);
        if (!$post || $post->user_id !== Auth::id() || !$this->canEdit($post)) {
            session()->flash('wall-error', 'مجوز ویرایش ندارید.');
            $this->cancelEdit();
            return;
        }
        $this->validate([
            'editingContent' => 'required|string|min:5|max:1000',
            'editingPostType' => 'required|in:status,report,milestone,issue,visit,knowledge',
            'editingAttachments' => 'nullable|array|max:5',
            'editingAttachments.*' => 'file|mimes:pdf,jpg,jpeg,png,zip,rar,doc,docx,xls,xlsx|max:5120',
        ]);
        $post->update([
            'content' => $this->editingContent,
            'type' => $this->editingPostType,
        ]);
        // Delete removed attachments
        foreach ($this->deletedAttachmentIds as $id) {
            $attachment = WallPostAttachment::find($id);
            if ($attachment) {
                Storage::disk('public')->delete($attachment->file_path);
                $attachment->delete();
            }
        }
        // Add new attachments
        if (!empty($this->editingAttachments)) {
            foreach ($this->editingAttachments as $file) {
                $path = $file->store('project_wall/' . $this->project->id, 'public');
                $post->attachments()->create([
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'file_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }
        session()->flash('wall-success', 'پست با موفقیت به‌روزرسانی شد.');
        $this->cancelEdit();
    }

    public function cancelEdit()
    {
        $this->editingPostId = null;
        $this->editingContent = '';
        $this->editingPostType = 'status';
        $this->editingAttachments = [];
        $this->deletedAttachmentIds = [];
    }

    public function deletePost(ProjectWallPost $post)
    {
        if (!$this->canDelete($post)) {
            session()->flash('wall-error', 'مهلت حذف این پست (۴۸ ساعت) به پایان رسیده است.');
            return;
        }
        if ($post->user_id === Auth::id()) {
            foreach ($post->attachments as $attachment) {
                Storage::disk('public')->delete($attachment->file_path);
            }
            $post->delete();
            session()->flash('wall-success', 'پست حذف شد.');
            $this->resetPage();
        } else {
            session()->flash('wall-error', 'شما فقط مجاز به حذف پست‌های خود هستید.');
        }
    }

    public function canEdit(ProjectWallPost $post): bool
    {
        return $post->user_id === Auth::id() && $post->created_at->diffInHours(now()) < 240;
    }

    public function canDelete(ProjectWallPost $post): bool
    {
        return $post->user_id === Auth::id() && $post->created_at->diffInHours(now()) < 48;
    }

    public function removeAttachment($key)
    {
        if (isset($this->attachments[$key])) {
            unset($this->attachments[$key]);
            $this->attachments = array_values($this->attachments);
        }
    }

    public function removeEditingAttachment($key)
    {
        if (isset($this->editingAttachments[$key])) {
            unset($this->editingAttachments[$key]);
            $this->editingAttachments = array_values($this->editingAttachments);
        }
    }

    public function removeExistingAttachment($id)
    {
        $this->deletedAttachmentIds[] = $id;
    }

    public function getPosts()
    {
        $query = ProjectWallPost::where('project_id', $this->project->id)
            ->with(['user:id,name,image', 'attachments', 'comments.user:id,name,image'])
            ->orderBy('created_at', 'desc');
        if ($this->filterType !== 'all') {
            $query->where('type', $this->filterType);
        }
        if ($this->searchQuery) {
            $query->where('content', 'like', '%' . $this->searchQuery . '%');
        }
        return $query->paginate(10);
    }

    public function loadAllComments(ProjectWallPost $post)
    {
        $this->currentPostComments = $post->comments()->with('user:id,name,image')->orderBy('created_at', 'asc')->get();
        $this->currentPostIdForComments = $post->id;
        $this->showAllCommentsModal = true;
    }

    public function closeCommentsModal()
    {
        $this->showAllCommentsModal = false;
        $this->currentPostComments = [];
        $this->currentPostIdForComments = null;
    }

    public function render()
    {
        return view('livewire.admin.projects.project-wall', [
            'posts' => $this->getPosts(),
        ])->layout('admin.master');
    }
}
