<?php

namespace App\Livewire\Admin\Projects\Schedule;

use App\Models\Project;
use App\Models\ProjectSchedule;
use Livewire\Attributes\Layout;
use Livewire\Component;

class ProjectScheduleCreate extends Component
{
    public $project_id;
    // 💡 متغیرها باید تاریخ میلادی را نگه دارند (نوع HTML Input Date)
    public $LOA_date, $contract_date, $actual_completion_date, $execution_duration;
    public $projects, $schedules;

    public $projectName;
    public $projectCode;

    protected $rules = [
        'project_id' => 'required|exists:projects,id',
        'LOA_date' => 'nullable|date',
        'contract_date' => 'nullable|date',
        'actual_completion_date' => 'nullable|date',
        'execution_duration' => 'required|integer|min:1',
    ];

    #[Layout('admin.master')]
    public function render()
    {
        // 💡 واکشی پروژه‌ها به صورت ساده برای رندر
        $this->projects = Project::select('id', 'name')->get();

        // 💡 واکشی زمان‌بندی‌ها با eager loading
        $this->schedules = ProjectSchedule::with('project:id,name,code')
            ->orderBy('id', 'desc')->get();

        return view('livewire.admin.projects.schedule.project-schedule-create', [
            'schedules' => $this->schedules,
        ]);
    }

    public function mount($projectId)
    {
        $this->project_id = $projectId;

        if ($project = Project::find($projectId)) {
            $this->projectName = $project->name;
            $this->projectCode = $project->code;
        } else {
            $this->projectName = 'پروژه ناشناس';
            $this->projectCode = '---';
        }

        // 💡 لود کردن تاریخ‌ها به صورت میلادی برای ورودی (Input Type="date")
        if ($schedule = ProjectSchedule::where('project_id', $projectId)->first()) {
            $this->LOA_date = $schedule->LOA_date ? \Carbon\Carbon::parse($schedule->LOA_date)->toDateString() : '';
            $this->contract_date = $schedule->contract_date ? \Carbon\Carbon::parse($schedule->contract_date)->toDateString() : '';
            $this->actual_completion_date = $schedule->actual_completion_date ? \Carbon\Carbon::parse($schedule->actual_completion_date)->toDateString() : '';
            $this->execution_duration = $schedule->execution_duration;
        } else {
            $this->LOA_date = '';
            $this->contract_date = '';
            $this->actual_completion_date = '';
            $this->execution_duration = '';
        }
    }

    public function saveSchedule()
    {
        $this->validate();

        // ذخیره یا به‌روزرسانی اطلاعات زمان‌بندی
        ProjectSchedule::updateOrCreate(
            ['project_id' => $this->project_id],
            [
                'LOA_date' => $this->LOA_date,
                'contract_date' => $this->contract_date,
                'actual_completion_date' => $this->actual_completion_date,
                'execution_duration' => $this->execution_duration,
            ]
        );

        session()->flash('message', 'اطلاعات زمان‌بندی با موفقیت ذخیره شد.');
    }

    // 💡 متد کمکی برای تبدیل تاریخ به شمسی جهت نمایش
    public function getPersianDate($date)
    {
        // فرض می‌شود تابع verta یا Jalali در سیستم شما تعریف شده است
        return $date ? (function_exists('verta') ? verta($date)->format('Y/m/d') : \Carbon\Carbon::parse($date)->format('Y-m-d')) : '—';
    }
}
