<?php

namespace App\Livewire\Admin\Purchase;

use App\Models\Project;
use App\Models\User;
use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestApproval;
use App\Notifications\PurchaseRequestStatusUpdated;
use App\Notifications\PurchaseRequestCreated;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\File; // برای رفع خطای تصویر ۵
use Barryvdh\DomPDF\Facade\Pdf; // در ابتدای فایل اضافه شود
use Livewire\Attributes\Computed;
use Mpdf\Mpdf;
use Symfony\Component\HttpFoundation\StreamedResponse;


class AllPurchaseRequests extends Component
{
    use WithPagination;

    // فیلترها
    public $search = '';
    public $statusFilter = '';
    public $projectFilter = '';
    public $userFilter = '';
    public $comment = '';
    public $selectedRequests = []; // حتماً به صورت آرایه خالی مقداردهی شود

    protected $paginationTheme = 'bootstrap';

    // ریست صفحه هنگام تغییر فیلترها
    public function updatingSearch() { $this->resetPage(); }
    public function updatingStatusFilter() { $this->resetPage(); }
    public function updatingProjectFilter() { $this->resetPage(); }
    public function updatingUserFilter() { $this->resetPage(); }

    public function resetFilters()
    {
        $this->reset(['search', 'statusFilter', 'projectFilter', 'userFilter']);
    }

    #[Computed]
    public function projects()
    {
        return Project::orderBy('name')->get();
    }

    #[Computed]
    public function users()
    {
        return User::whereHas('purchaseRequests')->orderBy('name')->get();
    }

    #[Computed]
    public function purchaseRequests()
    {
        return PurchaseRequest::with(['project', 'user', 'items.item', 'approvals.user', 'approvals.role'])
            ->when($this->search, function ($query) {
                $query->where('description', 'like', '%' . $this->search . '%')
                    ->orWhereHas('user', fn ($q) => $q->where('name', 'like', '%' . $this->search . '%'))
                    ->orWhereHas('project', fn ($q) => $q->where('name', 'like', '%' . $this->search . '%'));
            })
            ->when($this->statusFilter, fn($q) => $q->where('status', $this->statusFilter))
            ->when($this->projectFilter, fn($q) => $q->where('project_id', $this->projectFilter))
            ->when($this->userFilter, fn($q) => $q->where('user_id', $this->userFilter))
            ->orderBy('created_at', 'desc')
            ->paginate(12);
    }

    /**
     * تایید درخواست توسط مدیر جاری
     */
    public function approveRequest($requestId)
    {
        DB::transaction(function () use ($requestId) {
            $request = PurchaseRequest::findOrFail($requestId);

            // پیدا کردن ردیف تایید مربوط به این مدیر که هنوز تایید نکرده
            $approval = $request->approvals()
                ->where('user_id', Auth::id())
                ->whereNull('is_approved')
                ->first();

            if (!$approval) {
                $this->dispatch('alert', type: 'error', message: 'شما در صف تایید این درخواست نیستید یا قبلاً اقدام کرده‌اید.');
                return;
            }

            $approval->update([
                'is_approved' => true,
                'approved_at' => now(),
                'comment' => $this->comment,
            ]);

            $this->processNextStep($request);
            $this->reset(['comment']);
            $this->dispatch('alert', type: 'success', message: 'تایید شما با موفقیت ثبت شد.');
        });
    }

    /**
     * رد درخواست توسط مدیر
     */
    public function rejectRequest($requestId)
    {
        DB::transaction(function () use ($requestId) {
            $request = PurchaseRequest::findOrFail($requestId);
            $approval = $request->approvals()->where('user_id', Auth::id())->whereNull('is_approved')->first();

            if ($approval) {
                $approval->update([
                    'is_approved' => false,
                    'approved_at' => now(),
                    'comment' => $this->comment,
                ]);

                $request->update(['status' => 'rejected']);

                // اطلاع به متقاضی
                $request->user->notify(new PurchaseRequestStatusUpdated($request, 'rejected'));

                $this->reset(['comment']);
                $this->dispatch('alert', type: 'info', message: 'درخواست رد شد و به اطلاع متقاضی رسید.');
            }
        });
    }

    /**
     * مدیریت گام بعدی تایید یا اتمام فرآیند
     */
    protected function processNextStep($request)
    {
        $request->refresh();
        $totalApprovals = $request->approvals->count();
        $approvedCount = $request->approvals->where('is_approved', true)->count();

        if ($approvedCount === $totalApprovals) {
            // همه تایید کرده‌اند
            $request->update(['status' => 'approved']);
            $request->user->notify(new PurchaseRequestStatusUpdated($request, 'approved'));
        } else {
            // اطلاع‌رسانی به نفر بعدی در صف بر اساس اولویت
            $nextApprover = $request->approvals()
                ->whereNull('is_approved')
                ->orderBy('priority', 'asc')
                ->first();

            if ($nextApprover && $nextApprover->user) {
                $nextApprover->user->notify(new PurchaseRequestCreated($request));
            }
        }
    }

    /**
     * تایید دسته‌جمعی (Bulk Action)
     */
    public function bulkApprove($ids)
    {
        foreach ($ids as $id) {
            $this->approveRequest($id);
        }
        $this->dispatch('alert', type: 'success', message: count($ids) . ' درخواست با موفقیت تایید شدند.');
    }

    /**
     * خروجی ساده اکسل/CSV از لیست جاری
     */
    public function exportExcel()
    {
        $filename = "Purchase_Requests_Report_" . now()->format('Y-m-d') . ".csv";
        $headers = [
            "Content-type"        => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma"              => "no-cache",
            "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
            "Expires"             => "0"
        ];

        $callback = function() {
            $file = fopen('php://output', 'w');
            fputs($file, "\xEF\xBB\xBF"); // اضافه کردن BOM برای نمایش درست حروف فارسی در اکسل
            fputcsv($file, ['شناسه', 'پروژه', 'متقاضی', 'مبلغ کل (ریال)', 'وضعیت', 'تاریخ ثبت']);

            foreach ($this->purchaseRequests() as $row) {
                fputcsv($file, [
                    $row->id,
                    $row->project->name,
                    $row->user->name,
                    $row->total_price,
                    $row->status,
                    jdate($row->created_at)->format('Y/m/d')
                ]);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * حذف فیزیکی درخواست (فقط توسط ادمین کل)
     */
    public function deleteRequest($id)
    {
        $request = PurchaseRequest::findOrFail($id);


        DB::transaction(function () use ($request) {
            $request->items()->delete();
            $request->approvals()->delete();
            $request->delete();
        });

        $this->dispatch('alert', type: 'success', message: 'درخواست و تمامی سوابق مرتبط حذف شدند.');
    }


//    public function downloadPdf($requestId): StreamedResponse
//    {
//        // ۱. دریافت اطلاعات با روابط کامل
//        $request = \App\Models\PurchaseRequest::with(['project', 'user', 'items', 'approvals.user', 'approvals.role'])
//            ->findOrFail($requestId);
//// ۱. آماده‌سازی لوگوی اختصاصی شرکتِ متقاضی
//        $companyLogoBase64 = null;
//        $company = $request->user->company;
//
//        if ($company && $company->image) {
//            // مسیر فیزیکی عکس در پوشه storage
//            $logoPath = storage_path('app/company/logo/' . $company->image);
//
//            if (\File::exists($logoPath)) {
//                $logoData = \File::get($logoPath);
//                $extension = pathinfo($logoPath, PATHINFO_EXTENSION);
//                $companyLogoBase64 = 'data:image/' . $extension . ';base64,' . base64_encode($logoData);
//            }
//        }
//
//        // ۲. پیکربندی mPDF (مطابق با تنظیمات فونت شما)
//        $mpdfConfig = [
//            'mode' => 'utf-8',
//            'format' => 'A4',
//            'margin_left' => 15,
//            'margin_right' => 15,
//            'margin_top' => 15,
//            'margin_bottom' => 15,
//            'fontDir' => [public_path('panel/dist/fonts/')],
//            'fontdata' => [
//                'vazir' => ['R' => 'Vazir.ttf', 'B' => 'Vazir-Bold.ttf', 'useOTL' => 0xFF],
//                'peyda' => ['R' => 'Peyda-Regular.ttf', 'B' => 'Peyda-Bold.ttf', 'useOTL' => 0xFF],
//            ],
//            'default_font' => 'vazir',
//        ];
//
//        try {
//            $mpdf = new Mpdf($mpdfConfig);
//            $mpdf->SetDirectionality('rtl');
//
//            // ۳. آماده‌سازی امضاها به صورت Base64 (بسیار حیاتی برای نمایش در PDF)
//            $approvalsData = $request->approvals->sortBy('priority')->map(function ($app) {
//                $base64Sign = null;
//                if ($app->is_approved && $app->user->sign) {
//                    $path = storage_path('app/public/' . $app->user->sign);
//                    if (File::exists($path)) {
//                        $imgData = File::get($path);
//                        $ext = pathinfo($path, PATHINFO_EXTENSION);
//                        $base64Sign = 'data:image/' . $ext . ';base64,' . base64_encode($imgData);
//                    }
//                }
//                return [
//                    'role' => $app->role->name,
//                    'name' => $app->user->name,
//                    'date' => $app->approved_at ? jdate($app->approved_at)->format('Y/m/d H:i') : null,
//                    'signature' => $base64Sign,
//                    'is_approved' => $app->is_approved
//                ];
//            });
//
//            // ۴. رندر محتوای HTML
//            $html = view('livewire.admin.purchase.pdf-print', [
//                'request' => $request,
//                'companyLogo' => $companyLogoBase64, // لوگوی شرکت متقاضی
//                'approvals' => $approvalsData,
//            ])->render();
//
//            $mpdf->WriteHTML($html);
//
//            // ۵. خروجی به صورت StreamedResponse برای رفع خطا
//            $fileName = "Purchase-Order-PR-{$request->id}.pdf";
//
//            return response()->streamDownload(function () use ($mpdf) {
//                echo $mpdf->Output('', 'S');
//            }, $fileName, [
//                'Content-Type' => 'application/pdf',
//            ]);
//
//        } catch (\Exception $e) {
//            \Log::error("PDF Generation Error: " . $e->getMessage());
//            // در صورت خطا، کاربر را با پیام باخبر کنید (اختیاری)
//            throw $e;
//        }
//    }


    /**
     * آماده‌سازی داده‌های مربوط به تاییدات و تبدیل امضاها به Base64
     */
    private function prepareApprovalsData($request)
    {
        return $request->approvals->sortBy('priority')->map(function ($app) {
            $base64Sign = null;

            // اگر تایید شده بود و کاربر امضا داشت
            if ($app->is_approved && $app->user && $app->user->sign) {
                // مسیر فایل امضا در پوشه storage
                $path = storage_path('app/public/' . $app->user->sign);

                if (file_exists($path)) {
                    $imgData = file_get_contents($path);
                    $ext = pathinfo($path, PATHINFO_EXTENSION);
                    // تبدیل به رشته Base64 جهت نمایش قطعی در mPDF
                    $base64Sign = 'data:image/' . $ext . ';base64,' . base64_encode($imgData);
                }
            }

            return [
                'role' => $app->role->name ?? '---',
                'name' => $app->user->name ?? '---',
                'date' => $app->approved_at ? jdate($app->approved_at)->format('Y/m/d H:i') : null,
                'signature' => $base64Sign,
                'is_approved' => $app->is_approved
            ];
        });
    }

    public function downloadPdf($requestId)
    {
        $request = \App\Models\PurchaseRequest::with(['user.company', 'project', 'items', 'approvals.user'])->findOrFail($requestId);

        // ۱. آماده‌سازی لوگو به صورت Base64 برای تضمین نمایش در PDF
        $companyLogoBase64 = null;
        $filename = $request->user->company->image ?? null;

        if ($filename) {
            // مطابق مسیر شما در متد getCompanyLogoUrl، مسیر فیزیکی فایل را پیدا می‌کنیم
            // اگر فایل‌ها در public/app/images/company/logo هستند:
            $logoPath = public_path('app/images/company/logo/' . $filename);

            if (file_exists($logoPath)) {
                $logoData = file_get_contents($logoPath);
                $extension = pathinfo($logoPath, PATHINFO_EXTENSION);
                $companyLogoBase64 = 'data:image/' . $extension . ';base64,' . base64_encode($logoData);
            }
        }

        // ۲. تولید PDF با mPDF
        $mpdf = new \Mpdf\Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4',
            'default_font' => 'vazir'
        ]);

        $html = view('livewire.admin.purchase.pdf-print', [
            'request' => $request,
            'companyLogo' => $companyLogoBase64, // ارسال داده‌های Base64 به جای URL
            'approvals' => $this->prepareApprovalsData($request), // متدی که اطلاعات امضاها را آماده می‌کند
        ])->render();

        $mpdf->WriteHTML($html);
        return response()->streamDownload(fn() => print($mpdf->Output('', 'S')), "PR-{$request->id}.pdf");
    }

//    public function downloadPdf($requestId): StreamedResponse
//    {
//        $request = \App\Models\PurchaseRequest::with(['project', 'user.company', 'items', 'approvals.user', 'approvals.role'])
//            ->findOrFail($requestId);
//
//        // ۱. آماده‌سازی لوگوی شرکت با مسیر استاندارد لاراول
//        $companyLogoBase64 = null;
//        $company = $request->user->company;
//
//        if ($company && $company->image) {
//            // تغییر مهم: استفاده از Storage::path برای پیدا کردن مسیر دقیق فیزیکی
//            // فرض بر این است که در دیسک 'public' ذخیره کرده‌اید
//            $logoPath = storage_path('app/public/logo/' . $company->image);
//
//            // اگر از دیسک اختصاصی استفاده می‌کنید، این خط را جایگزین کنید:
//            // $logoPath = \Storage::disk('company')->path('logo/' . $company->image);
//
//            if (file_exists($logoPath)) {
//                $logoData = file_get_contents($logoPath);
//                $extension = pathinfo($logoPath, PATHINFO_EXTENSION);
//                $companyLogoBase64 = 'data:image/' . $extension . ';base64,' . base64_encode($logoData);
//            }
//        }
//
//        $mpdfConfig = [
//            'mode' => 'utf-8',
//            'format' => 'A4',
//            'margin_left' => 10,
//            'margin_right' => 10,
//            'margin_top' => 10,
//            'margin_bottom' => 10,
//            'fontDir' => [public_path('panel/dist/fonts/')],
//            'fontdata' => [
//                'vazir' => ['R' => 'Vazir.ttf', 'B' => 'Vazir-Bold.ttf', 'useOTL' => 0xFF],
//            ],
//            'default_font' => 'vazir',
//            // تغییر مهم: افزایش محدودیت حافظه برای پردازش تصاویر Base64
//            'img_max_width' => 0,
//            'img_max_height' => 0,
//        ];
//
//        try {
//            $mpdf = new \Mpdf\Mpdf($mpdfConfig);
//            $mpdf->SetDirectionality('rtl');
//
//            // تغییر مهم: اجازه دادن به mPDF برای خواندن منابع محلی
//            $mpdf->showImageErrors = true;
//
//            $approvalsData = $request->approvals->sortBy('priority')->map(function ($app) {
//                $base64Sign = null;
//                if ($app->is_approved && $app->user->sign) {
//                    // دقت کنید: مسیر امضای کاربر معمولاً در پوشه storage/app/public است
//                    $path = storage_path('app/public/' . $app->user->sign);
//
//                    if (file_exists($path)) {
//                        $imgData = file_get_contents($path);
//                        $ext = pathinfo($path, PATHINFO_EXTENSION);
//                        $base64Sign = 'data:image/' . $ext . ';base64,' . base64_encode($imgData);
//                    }
//                }
//                return [
//                    'role' => $app->role->name ?? '---',
//                    'name' => $app->user->name ?? '---',
//                    'date' => $app->approved_at ? jdate($app->approved_at)->format('Y/m/d H:i') : null,
//                    'signature' => $base64Sign,
//                    'is_approved' => $app->is_approved
//                ];
//            });
//
//            $html = view('livewire.admin.purchase.pdf-print', [
//                'request' => $request,
//                'companyLogo' => $companyLogoBase64,
//                'approvals' => $approvalsData,
//            ])->render();
//
//            $mpdf->WriteHTML($html);
//
//            return response()->streamDownload(function () use ($mpdf) {
//                echo $mpdf->Output('', 'S');
//            }, "PR-{$request->id}.pdf");
//
//        } catch (\Exception $e) {
//            \Log::error("PDF Error: " . $e->getMessage());
//            throw $e;
//        }
//    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.purchase.all-purchase-requests');
    }
}
