<?php

namespace App\Livewire\Admin\Purchase;

use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestApproval;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use Illuminate\Database\Eloquent\Builder;

class ApprovalInbox extends Component
{
    use WithPagination;

    public $comment = [];
    public $search = '';
    public $selectedRequest = null;
    public $activeTab = 'pending';
    protected $paginationTheme = 'bootstrap';

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedActiveTab()
    {
        $this->resetPage();
    }

    #[Computed]
    public function pendingRequests()
    {
        $userId = Auth::id();

        return PurchaseRequest::withSum('items as total_amount', DB::raw('quantity * price'))
            ->with(['project', 'user', 'items', 'approvals.user', 'approvals.role'])
            ->whereHas('approvals', function (Builder $query) use ($userId) {
                $query->where('user_id', $userId)->whereNull('is_approved');
            })
            ->whereDoesntHave('approvals', function (Builder $query) use ($userId) {
                $query->where('priority', '<', function ($sub) use ($userId) {
                    $sub->select('priority')
                        ->from('purchase_request_approvals as pra_inner')
                        ->where('pra_inner.user_id', $userId)
                        ->whereColumn('pra_inner.purchase_request_id', 'purchase_request_approvals.purchase_request_id')
                        ->limit(1);
                })->whereNull('is_approved');
            })
            ->when($this->search, function ($query) {
                $query->where('description', 'like', "%{$this->search}%")
                    ->orWhereHas('project', fn($q) => $q->where('name', 'like', "%{$this->search}%"));
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10, ['*'], 'pendingPage');
    }

    #[Computed]
    public function historyRequests()
    {
        $userId = Auth::id();

        return PurchaseRequest::withSum('items as total_amount', DB::raw('quantity * price'))
            ->with([
                'project',
                'user',
                'approvals' => fn($q) => $q->where('user_id', $userId)->whereNotNull('is_approved')
            ])
            ->whereHas('approvals', function (Builder $query) use ($userId) {
                $query->where('user_id', $userId)->whereNotNull('is_approved');
            })
            ->when($this->search, fn($query) => $query->where('description', 'like', "%{$this->search}%"))
            ->orderBy('updated_at', 'desc')
            ->paginate(10, ['*'], 'historyPage');
    }

    public function showDetails($requestId)
    {
        $this->selectedRequest = PurchaseRequest::with([
            'project', 'user', 'items', 'approvals.user', 'approvals.role', 'costCenter'
        ])->findOrFail($requestId);
    }

    public function closeDetailModal()
    {
        $this->selectedRequest = null;
        $this->comment = [];
    }

    public function approveRequest($requestId)
    {
        $this->processApproval($requestId, true);
    }

    public function rejectRequest($requestId)
    {
        $this->processApproval($requestId, false);
    }

    protected function processApproval($requestId, $isApproved)
    {
        $request = PurchaseRequest::findOrFail($requestId);
        $userId = Auth::id();
        $commentText = $this->comment[$requestId] ?? null;

        $approval = $request->approvals()
            ->where('user_id', $userId)
            ->whereNull('is_approved')
            ->orderBy('priority')
            ->first();

        if (!$approval) {
            session()->flash('error', 'شما مجاز به تأیید این درخواست نیستید.');
            return;
        }

        $minPendingPriority = $request->approvals()->whereNull('is_approved')->min('priority');
        if ($approval->priority !== $minPendingPriority) {
            session()->flash('error', 'هنوز نوبت شما برای تأیید این درخواست نرسیده است.');
            return;
        }

        $approval->update([
            'is_approved' => $isApproved,
            'approved_at' => now(),
            'comment' => $commentText,
        ]);

        $this->checkRequestStatus($request);

        unset($this->comment[$requestId]);
        $this->selectedRequest = null;
        $this->resetPage();

        session()->flash('message', 'درخواست خرید با موفقیت ' . ($isApproved ? 'تأیید' : 'رد') . ' شد.');
    }

    protected function checkRequestStatus(PurchaseRequest $request)
    {
        $approvals = $request->approvals;

        if ($approvals->where('is_approved', false)->isNotEmpty()) {
            $request->update(['status' => 'rejected']);
            return;
        }

        if ($approvals->whereNull('is_approved')->isEmpty()) {
            $request->update(['status' => 'approved']);
        }
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.purchase.approval-inbox');
    }
}
