<?php

namespace App\Livewire\Admin\Purchase;

use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestApproval;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use Illuminate\Database\Eloquent\Builder;

class ApprovePurchaseRequests extends Component
{
    use WithPagination;

    public $comment = [];
    public $search = '';
    protected $paginationTheme = 'bootstrap';

    public function updatingSearch() { $this->resetPage(); }

    #[Computed]
    public function pendingRequests()
    {
        $userId = Auth::id();
        $nextPriority = PurchaseRequestApproval::whereNull('is_approved')->min('priority');

        if (is_null($nextPriority)) {
            return PurchaseRequest::where('id', 0)->paginate(10);
        }

        // 💡 اصلاح کوئری: فقط Eager Loading می‌کنیم و محاسبه را به Accessor مدل می‌سپاریم.
        return PurchaseRequest::with(['project', 'user', 'items', 'approvals.user', 'approvals.role'])
            ->whereHas('approvals', function (Builder $query) use ($userId, $nextPriority) {
                $query->where('user_id', $userId)
                    ->where('priority', $nextPriority)
                    ->whereNull('is_approved');
            })
            ->when($this->search, function ($query) {
                $query->where('description', 'like', '%' . $this->search . '%')
                    ->orWhereHas('project', fn($q) => $q->where('name', 'like', '%' . $this->search . '%'));
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);
    }
    // --- منطق تأیید و رد (بدون تغییر) ---

    public function approveRequest($requestId)
    {
        $this->processApproval($requestId, true);
    }

    public function rejectRequest($requestId)
    {
        $this->processApproval($requestId, false);
    }

    protected function processApproval($requestId, $isApproved)
    {
        $request = PurchaseRequest::findOrFail($requestId);
        $userId = Auth::id();
        $commentText = $this->comment[$requestId] ?? null;

        $approval = $request->approvals()
            ->where('user_id', $userId)
            ->whereNull('is_approved')
            ->orderBy('priority')
            ->first();

        if (!$approval || $approval->priority > $request->approvals()->whereNull('is_approved')->min('priority')) {
            session()->flash('error', 'شما مجاز به تأیید این درخواست نیستید یا نوبت تأیید شما نیست.');
            return;
        }

        $approval->update([
            'is_approved' => $isApproved,
            'approved_at' => now(),
            'comment' => $commentText,
        ]);

        $this->checkRequestStatus($request);

        unset($this->comment[$requestId]);
        session()->flash('message', 'درخواست خرید با موفقیت ' . ($isApproved ? 'تأیید' : 'رد') . ' شد.');
    }

    protected function checkRequestStatus(PurchaseRequest $request)
    {
        $approvals = $request->approvals;

        if ($approvals->where('is_approved', false)->isNotEmpty()) {
            $request->update(['status' => 'rejected']);
            return;
        }

        if ($approvals->whereNull('is_approved')->isEmpty()) {
            $request->update(['status' => 'approved']);
            return;
        }
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.purchase.approve-purchase-requests');
    }
}
