<?php

namespace App\Livewire\Admin\Purchase;

use Livewire\Attributes\Layout;
use Livewire\Component;
use App\Models\PurchaseRequest;
use App\Models\Project;
use App\Models\Item;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Livewire\Attributes\On;

class EditPurchaseRequest extends Component
{
    public $requestId;
    public $project_id;
    public $description;
    public $items = []; // [ ['id' => 1, 'item_id'=>..., 'quantity'=>..., 'price'=>..., 'is_service' => bool], ... ]
    public $status = '';
    public $projectBudget = null;

    protected $rules = [
        'project_id' => 'required|exists:projects,id',
        'description' => 'required|string',
        'items' => 'required|array|min:1',
        'items.*.item_id' => 'nullable|exists:items,id', // 💡 nullable برای خدمات
        'items.*.quantity' => 'required|integer|min:1',
        'items.*.price' => 'required|numeric|min:0',
        'items.*.id' => 'nullable|integer',
        // 💡 Validation: در صورت نبودن item_id، باید description (نام خدمت) پر شود
        'items.*.service_description' => 'required_if:items.*.item_id,null|nullable|string|max:500',
    ];

    // --- Computed Properties ---

    #[Computed]
    public function availableItems()
    {
        return Item::all();
    }

    #[Computed]
    public function projects()
    {
        return Project::all();
    }

    #[Computed]
    public function totalAmount()
    {
        return collect($this->items)->sum(function($i) {
            return (float)($i['quantity'] ?? 0) * (float)($i['price'] ?? 0);
        });
    }

    #[Computed]
    public function totalCount()
    {
        return collect($this->items)->sum(function($i) {
            return (int)($i['quantity'] ?? 0);
        });
    }

    #[Layout('admin.master')]
    public function mount($id)
    {
        $this->requestId = $id;
        $purchaseRequest = PurchaseRequest::with(['items', 'items.item', 'project'])->findOrFail($id);

        if ($purchaseRequest->user_id !== Auth::id() || $purchaseRequest->status !== 'pending') {
            abort(403, 'شما مجاز به ویرایش این درخواست نیستید.');
        }

        $this->project_id = $purchaseRequest->project_id;
        $this->description = $purchaseRequest->description;
        $this->status = $purchaseRequest->status;

        // 💡 اصلاح: لود کردن داده‌ها از مدل PurchaseRequestItem (HasMany)
        $this->items = $purchaseRequest->items->map(function ($item) {
            $isService = is_null($item->item_id);
            return [
                'id' => $item->id,
                'item_id' => $item->item_id,
                'quantity' => $item->quantity,
                'price' => $item->price,
                // 💡 برای خدمات، شرح را از ستون description لود کن
                'service_description' => $isService ? $item->description : null,
                'is_service' => $isService,
            ];
        })->toArray();

        $this->projectBudget = $purchaseRequest->project->budget ?? null;
    }

    public function addItemRow()
    {
        // 💡 افزودن فیلدهای لازم برای آیتم جدید
        $this->items[] = [
            'id' => null,
            'item_id' => null,
            'quantity' => 1,
            'price' => 0,
            'service_description' => null,
            'is_service' => false,
        ];
    }

    public function removeItemRow($index)
    {
        unset($this->items[$index]);
        $this->items = array_values($this->items);
    }

    public function updateRequest()
    {
        $this->validate();

        $purchaseRequest = PurchaseRequest::findOrFail($this->requestId);

        if ($purchaseRequest->user_id !== Auth::id() || $purchaseRequest->status !== 'pending') {
            session()->flash('message', 'شما مجاز به ویرایش این درخواست نیستید.');
            return;
        }

        $purchaseRequest->update([
            'project_id' => $this->project_id,
            'description' => $this->description,
        ]);

        $currentIds = collect($this->items)->pluck('id')->filter()->all();

        // 1. حذف آیتم‌هایی که از فرم حذف شده‌اند
        $purchaseRequest->items()
            ->whereNotIn('id', $currentIds)
            ->delete();

        // 2. به‌روزرسانی آیتم‌های موجود و ایجاد آیتم‌های جدید
        foreach ($this->items as $item) {
            // تعیین داده‌های ذخیره‌سازی
            $isService = empty($item['item_id']); // اگر item_id خالی باشد، خدمات است

            $data = [
                'item_id' => $item['item_id'],
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                // 💡 ذخیره شرح خدمات در صورت لزوم
                'description' => $isService ? $item['service_description'] : null,
            ];

            if ($item['id']) {
                // به‌روزرسانی آیتم موجود
                $purchaseRequest->items()->where('id', $item['id'])->update($data);
            } else {
                // ایجاد آیتم جدید
                $purchaseRequest->items()->create($data);
            }
        }

        session()->flash('message', 'درخواست با موفقیت ویرایش شد.');
        $this->dispatch('refresh-component'); // برای اطمینان از به‌روزرسانی لیست آیتم‌ها
    }

    public function updatedProjectId()
    {
        $project = Project::find($this->project_id);
        $this->projectBudget = $project ? $project->budget : null;
    }

    public function render()
    {
        return view('livewire.admin.purchase.edit-purchase-request');
    }
}
