<?php

namespace App\Livewire\Admin\Purchase;

use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\PurchaseRequest;
use App\Models\Project;
use Livewire\Attributes\Computed;

class MyPurchaseRequests extends Component
{
    use WithPagination;
    protected $paginationTheme = 'bootstrap';

    public $search = '';
    public $statusFilter = '';
    public $projectFilter = '';
    public $viewMode = 'grid';

    // ریست فیلترها
    public function resetFilters()
    {
        $this->reset(['search', 'statusFilter', 'projectFilter']);
        $this->dispatch('alert', type: 'info', message: 'فیلترها بازنشانی شدند.');
    }

    // آمار سریع برای داشبورد بالای صفحه
    #[Computed]
    public function stats()
    {
        $baseQuery = PurchaseRequest::where('user_id', Auth::id());
        return [
            'total' => $baseQuery->count(),
            'pending' => $baseQuery->clone()->where('status', 'pending')->count(),
            'approved' => $baseQuery->clone()->where('status', 'approved')->count(),
            'total_budget' => $baseQuery->clone()->where('status', 'approved')->get()->sum('total_price'),
        ];
    }

    #[Computed]
    public function projects()
    {
        return Project::orderBy('name')->get();
    }

    public function setViewMode($mode)
    {
        $this->viewMode = $mode;
    }

    #[Layout('admin.master')]
    public function render()
    {
        $myRequests = PurchaseRequest::with([
            'project',
            'items.item',
            'approvals.user',
            'approvals.role',
            'costCenter'
        ])
            ->where('user_id', Auth::id())
            ->when($this->search, function($q){
                $q->where(function($sub){
                    $sub->where('description', 'like', '%' . $this->search . '%')
                        ->orWhere('id', 'like', '%' . $this->search . '%')
                        ->orWhereHas('project', fn($p) => $p->where('name', 'like', '%' . $this->search . '%'));
                });
            })
            ->when($this->statusFilter, fn($q) => $q->where('status', $this->statusFilter))
            ->when($this->projectFilter, fn($q) => $q->where('project_id', $this->projectFilter))
            ->orderByDesc('created_at')
            ->paginate($this->viewMode == 'grid' ? 12 : 20);

        return view('livewire.admin.purchase.my-purchase-requests', [
            'myRequests' => $myRequests,
        ]);
    }

    public function deleteRequest($id)
    {
        $req = PurchaseRequest::findOrFail($id);

        if($req->user_id == Auth::id() && $req->status == 'pending'){
            DB::transaction(function() use ($req) {
                $req->items()->delete();
                $req->approvals()->delete();
                $req->delete();
            });
            $this->dispatch('alert', type: 'success', message: 'درخواست با موفقیت حذف شد.');
        } else {
            $this->dispatch('alert', type: 'error', message: 'حذف فقط برای درخواست‌های «در انتظار» مجاز است.');
        }
    }
}
