<?php

namespace App\Livewire\Admin\Purchase;

use App\Models\Payee;
use App\Models\PaymentRequest;
use Livewire\Attributes\{Computed, Layout, Url};
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('admin.master')]
class PayeeList extends Component
{
    use WithPagination;

    #[Url(history: true)]
    public $search = '';

    #[Url(history: true)]
    public $filter_type = '';

    #[Url(history: true)]
    public $sortField = 'total_paid'; // فیلد پیش‌فرض برای سورت

    #[Url(history: true)]
    public $sortDirection = 'desc';

    public $selectedPayeeId = null;
    public $payeeHistory = null;

    // ایده جدید: متد سورت سریع
    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'desc';
        }
    }

    #[Computed]
    public function stats()
    {
        return [
            'total_count' => Payee::count(),
            'total_paid' => PaymentRequest::where('status', 'paid')->sum('amount'),
            'pending_amount' => PaymentRequest::where('status', 'pending')->sum('amount'), // ایده جدید: مبلغ در جریان
            'recent_active' => Payee::whereHas('paymentRequests', fn($q) => $q->where('created_at', '>=', now()->subDays(30)))->count(),
        ];
    }

    #[Computed]
    public function payees()
    {
        return Payee::query()
            ->withSum(['paymentRequests as total_paid' => fn($q) => $q->where('status', 'paid')], 'amount')
            ->withCount(['paymentRequests as pending_requests' => fn($q) => $q->where('status', 'pending')])
            ->when($this->filter_type, fn ($query) => $query->where('type', $this->filter_type))
            ->when($this->search, fn ($query) => $query->where(fn($q) =>
            $q->where('name', 'like', '%' . $this->search . '%')
                ->orWhere('national_id', 'like', '%' . $this->search . '%')
            ))
            // سورت هوشمند بر اساس انتخاب کاربر
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate(12);
    }

    public function showFinancialHistory($payeeId)
    {
        $this->selectedPayeeId = $payeeId;
        $this->payeeHistory = Payee::with(['paymentRequests' => fn($q) => $q->with('project')->latest()])
            ->withSum(['paymentRequests as grand_total' => fn($q) => $q->where('status', 'paid')], 'amount')
            ->findOrFail($payeeId);

        $this->dispatch('open-history-modal');
    }

    public function closeHistoryModal()
    {
        $this->reset(['selectedPayeeId', 'payeeHistory']);
    }

    public function render()
    {
        return view('livewire.admin.purchase.payee-list');
    }
}
