<?php

namespace App\Livewire\Admin\Purchase;

use App\Models\Payee;
use Livewire\Attributes\{Layout, Computed, Url};
use Livewire\{Component, WithPagination};
use Illuminate\Support\Facades\DB;

#[Layout('admin.master')]
class PayeeManagement extends Component
{
    use WithPagination;

    public $payeeId;
    public $type = 'company';
    public $name, $national_id, $bank_name, $account_number, $sheba, $contact_person;

    #[Url(history: true)]
    public $search = '';

    protected $paginationTheme = 'bootstrap';

    /**
     * نگاشت پیش‌شماره‌های شبا به کدهای بانکی سیستم
     */
    private $shebaMap = [
        '010' => 'bmi', '011' => 'en', '012' => 'mellat', '013' => 'refah',
        '014' => 'tejarat', '015' => 'sepah', '016' => 'sarmayeh', '017' => 'msbi',
        '018' => 'bsi', '019' => 'pasargad', '021' => 'post', '054' => 'parsian',
        '056' => 'saman', '061' => 'shahr', '062' => 'ayandeh'
    ];

    #[Computed]
    public function stats()
    {
        return [
            'total' => Payee::count(),
            'company' => Payee::where('type', 'company')->count(),
            'person' => Payee::where('type', 'person')->count(),
            'recent' => Payee::where('created_at', '>=', now()->subDays(7))->count(),
        ];
    }

    #[Computed]
    public function iranianBanks()
    {
        return [
            'bmi' => ['name' => 'بانک ملی ایران', 'color' => '#004b93', 'logo' => 'bmi.png'],
            'sepah' => ['name' => 'بانک سپه', 'color' => '#eebf00', 'logo' => 'sepah.png'],
            'bsi' => ['name' => 'بانک صادرات ایران', 'color' => '#1b4897', 'logo' => 'bsi.png'],
            'mellat' => ['name' => 'بانک ملت', 'color' => '#cb0000', 'logo' => 'mellat.png'],
            'tejarat' => ['name' => 'بانک تجارت', 'color' => '#007cc2', 'logo' => 'tejarat.png'],
            'ayandeh' => ['name' => 'بانک آینده', 'color' => '#7a2022', 'logo' => 'ayandeh.png'],
            'pasargad' => ['name' => 'بانک پاسارگاد', 'color' => '#ca8a04', 'logo' => 'pasargad.png'],
            'parsian' => ['name' => 'بانک پارسیان', 'color' => '#9c282a', 'logo' => 'parsian.png'],
            'saman' => ['name' => 'بانک سامان', 'color' => '#5eade2', 'logo' => 'saman.png'],
            'shahr' => ['name' => 'بانک شهر', 'color' => '#e01b22', 'logo' => 'shahr.png'],
            'en' => ['name' => 'بانک اقتصاد نوین', 'color' => '#65318e', 'logo' => 'en.png'],
            'post' => ['name' => 'پست بانک ایران', 'color' => '#008444', 'logo' => 'post.png'],
        ];
    }

    protected function rules()
    {
        return [
            'type' => 'required|in:person,company',
            'name' => 'required|string|min:3|max:255',
            'national_id' => 'required|string|max:50|unique:payees,national_id,' . $this->payeeId,
            'bank_name' => 'required|string',
            'account_number' => 'nullable|numeric|digits_between:5,20',
            'sheba' => 'nullable|regex:/^([0-9]{24})$/',
            'contact_person' => 'nullable|string|max:100',
        ];
    }

    /**
     * هوشمندی تشخیص بانک هنگام تغییر شماره شبا
     */
    public function updatedSheba($value)
    {
        $value = preg_replace('/\D/', '', $value);
        $this->sheba = $value;

        if (strlen($value) >= 5) {
            $prefix = substr($value, 2, 3);
            if (isset($this->shebaMap[$prefix])) {
                $this->bank_name = $this->shebaMap[$prefix];
            }
        }
    }

    public function savePayee()
    {
        $this->validate();

        $data = [
            'type' => $this->type,
            'name' => trim($this->name),
            'national_id' => preg_replace('/\D/', '', $this->national_id),
            'bank_name' => $this->bank_name,
            'account_number' => $this->account_number,
            'sheba' => $this->sheba,
            'contact_person' => $this->type === 'company' ? $this->contact_person : null,
        ];

        DB::transaction(function () use ($data) {
            Payee::updateOrCreate(['id' => $this->payeeId], $data);
        });

        $this->dispatch('alert', [
            'type' => 'success',
            'message' => $this->payeeId ? 'پرونده ذینفع به‌روزرسانی شد' : 'ذینفع جدید با موفقیت ثبت شد'
        ]);

        $this->resetForm();
    }

    public function editPayee($id)
    {
        $payee = Payee::findOrFail($id);
        $this->payeeId = $payee->id;
        $this->fill($payee->toArray());
        $this->dispatch('scroll-to-top'); // فراخوانی اسکریپت سمت کلاینت
    }

    public function deletePayee($id)
    {
        $payee = Payee::findOrFail($id);
        if ($payee->paymentRequests()->exists()) {
            $this->dispatch('alert', ['type' => 'error', 'message' => 'این ذینفع دارای تراکنش مالی است و حذف نمی‌شود']);
            return;
        }
        $payee->delete();
        $this->dispatch('alert', ['type' => 'info', 'message' => 'رکورد مورد نظر حذف شد']);
    }

    public function resetForm()
    {
        $this->reset(['payeeId', 'name', 'national_id', 'bank_name', 'account_number', 'sheba', 'contact_person']);
        $this->resetValidation();
    }

    #[Computed]
    public function payeesList()
    {
        return Payee::query()
            ->when($this->search, function($q) {
                $q->where('name', 'like', "%{$this->search}%")
                    ->orWhere('national_id', 'like', "%{$this->search}%")
                    ->orWhere('sheba', 'like', "%{$this->search}%");
            })
            ->orderByDesc('created_at')
            ->paginate(12);
    }

    public function render() { return view('livewire.admin.purchase.payee-management'); }
}
