<?php

namespace App\Livewire\Admin\Purchase;

use Livewire\Attributes\Layout;
use Livewire\Component;
use App\Models\Project;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate;
use Illuminate\Database\Eloquent\Collection; // 💡 برای تایپ هینت بهتر

class ProjectBudgetManagement extends Component
{
    // --- فیلدهای Modal ---
    public $editProjectId;
    #[Validate('required|numeric|min:0', message: 'مبلغ بودجه نمی‌تواند خالی یا منفی باشد.')]
    public $newBudgetAmount;

    #[Layout('admin.master')]
    public function render()
    {
        // 💡 لود کردن روابط لازم برای محاسبه Accessorها
        // فرض می‌کنیم Accessorهای budget و total_spent در مدل Project تعریف شده‌اند
        $projects = Project::with([
            'purchaseRequests',
            'purchaseRequests.items',
            'mainContract.addenda',
            'details'
        ])->get();

        // 💡 آماده‌سازی داده‌ها برای نمایش
        $projectData = $projects->map(function ($project) {

            $budget = $project->budget;
            $spent = $project->total_spent;

            $remaining = $budget - $spent;
            $percent = $budget > 0 ? round(($spent / $budget) * 100) : 0;

            return (object)[
                'id' => $project->id,
                'name' => $project->name,
                'budget' => $budget,
                'spent' => $spent,
                'remaining' => $remaining,
                'percent' => $percent,
                // تعیین کلاس وضعیت برای خطاهای بودجه
                'status_class' => $spent > $budget ? 'budget-exceeded' : ($percent >= 80 ? 'budget-warning' : ''),
            ];
        });

        return view('livewire.admin.purchase.project-budget-management', [
            'projectData' => $projectData,
        ]);
    }

    // --- منطق ویرایش بودجه ---

    public function editBudget($projectId, $currentBudget)
    {
        $this->editProjectId = $projectId;
        $this->newBudgetAmount = $currentBudget;
        $this->dispatch('open-budget-modal');
    }

    public function saveBudget()
    {
        $this->validate();

        // 💡 منطق به‌روزرسانی بودجه باید بسته به محل ذخیره‌ی آن انجام شود.
        // اگر بودجه در ProjectFinancialDetail ذخیره می‌شود:
        // ProjectFinancialDetail::updateOrCreate(['project_id' => $this->editProjectId], ['contract_value' => $this->newBudgetAmount]);

        // موقتاً برای دموی Livewire:
        // $project = Project::findOrFail($this->editProjectId);
        // $project->update(['budget' => $this->newBudgetAmount]); // فرض می‌کنیم budget ستون موقتی دارد

        $this->reset(['editProjectId', 'newBudgetAmount']);
        session()->flash('message', 'بودجه پروژه با موفقیت به‌روزرسانی شد.');
    }
}
