<?php

namespace App\Livewire\Admin\Purchase;

use App\Models\Project;
use App\Models\User;
use App\Models\PurchaseRequest;
use App\Notifications\PurchaseRequestStatusUpdated;
use App\Notifications\PurchaseRequestCreated;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use Mpdf\Mpdf;
use Symfony\Component\HttpFoundation\StreamedResponse;

class ProjectPurchaseRequests extends Component
{
    use WithPagination;

    public $projectId;
    public $search = '';
    public $statusFilter = '';
    public $userFilter = '';
    public $comment = '';
    public $selectedRequests = [];

    protected $paginationTheme = 'bootstrap';

    public function mount($projectId)
    {
        $this->projectId = $projectId;
    }

    public function updatingSearch() { $this->resetPage(); }
    public function updatingStatusFilter() { $this->resetPage(); }
    public function updatingUserFilter() { $this->resetPage(); }

    public function resetFilters()
    {
        $this->reset(['search', 'statusFilter', 'userFilter', 'comment']);
    }

    #[Computed]
    public function project()
    {
        return Project::findOrFail($this->projectId);
    }

    #[Computed]
    public function users()
    {
        return User::whereHas('purchaseRequests', function($q) {
            $q->where('project_id', $this->projectId);
        })->orderBy('name')->get();
    }

    #[Computed]
    public function purchaseRequests()
    {
        return PurchaseRequest::with(['project', 'user', 'items.item', 'approvals.user', 'approvals.role'])
            ->where('project_id', $this->projectId)
            ->when($this->search, function ($query) {
                $query->where(function($sub) {
                    $sub->where('description', 'like', '%' . $this->search . '%')
                        ->orWhereHas('user', fn ($q) => $q->where('name', 'like', '%' . $this->search . '%'))
                        ->orWhere('id', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter, fn($q) => $q->where('status', $this->statusFilter))
            ->when($this->userFilter, fn($q) => $q->where('user_id', $this->userFilter))
            ->orderBy('created_at', 'desc')
            ->paginate(12);
    }

    public function approveRequest($requestId)
    {
        DB::transaction(function () use ($requestId) {
            $request = PurchaseRequest::findOrFail($requestId);
            $approval = $request->approvals()->where('user_id', Auth::id())->whereNull('is_approved')->first();

            if (!$approval) {
                $this->dispatch('alert', type: 'error', message: 'شما مجاز به تایید این مرحله نیستید.');
                return;
            }

            $approval->update(['is_approved' => true, 'approved_at' => now(), 'comment' => $this->comment]);
            $this->processNextStep($request);
            $this->reset(['comment']);
            $this->dispatch('alert', type: 'success', message: 'تایید با موفقیت ثبت شد.');
        });
    }

    public function rejectRequest($requestId)
    {
        DB::transaction(function () use ($requestId) {
            $request = PurchaseRequest::findOrFail($requestId);
            $approval = $request->approvals()->where('user_id', Auth::id())->whereNull('is_approved')->first();

            if ($approval) {
                $approval->update(['is_approved' => false, 'approved_at' => now(), 'comment' => $this->comment]);
                $request->update(['status' => 'rejected']);
                $request->user->notify(new PurchaseRequestStatusUpdated($request, 'rejected'));
                $this->reset(['comment']);
                $this->dispatch('alert', type: 'info', message: 'درخواست رد شد.');
            }
        });
    }

    protected function processNextStep($request)
    {
        $request->refresh();
        if ($request->approvals->where('is_approved', true)->count() === $request->approvals->count()) {
            $request->update(['status' => 'approved']);
            $request->user->notify(new PurchaseRequestStatusUpdated($request, 'approved'));
        } else {
            $next = $request->approvals()->whereNull('is_approved')->orderBy('priority', 'asc')->first();
            if ($next && $next->user) { $next->user->notify(new PurchaseRequestCreated($request)); }
        }
    }
    private function prepareApprovalsData($request)
    {
        return $request->approvals->sortBy('priority')->map(function ($app) {
            $base64Sign = null;

            // اگر تایید شده بود و کاربر امضا داشت
            if ($app->is_approved && $app->user && $app->user->sign) {
                // مسیر فایل امضا در پوشه storage
                $path = storage_path('app/public/' . $app->user->sign);

                if (file_exists($path)) {
                    $imgData = file_get_contents($path);
                    $ext = pathinfo($path, PATHINFO_EXTENSION);
                    // تبدیل به رشته Base64 جهت نمایش قطعی در mPDF
                    $base64Sign = 'data:image/' . $ext . ';base64,' . base64_encode($imgData);
                }
            }

            return [
                'role' => $app->role->name ?? '---',
                'name' => $app->user->name ?? '---',
                'date' => $app->approved_at ? jdate($app->approved_at)->format('Y/m/d H:i') : null,
                'signature' => $base64Sign,
                'is_approved' => $app->is_approved
            ];
        });
    }

    public function downloadPdf($requestId)
    {
        $request = \App\Models\PurchaseRequest::with(['user.company', 'project', 'items', 'approvals.user'])->findOrFail($requestId);

        // ۱. آماده‌سازی لوگو به صورت Base64 برای تضمین نمایش در PDF
        $companyLogoBase64 = null;
        $filename = $request->user->company->image ?? null;

        if ($filename) {
            // مطابق مسیر شما در متد getCompanyLogoUrl، مسیر فیزیکی فایل را پیدا می‌کنیم
            // اگر فایل‌ها در public/app/images/company/logo هستند:
            $logoPath = public_path('app/images/company/logo/' . $filename);

            if (file_exists($logoPath)) {
                $logoData = file_get_contents($logoPath);
                $extension = pathinfo($logoPath, PATHINFO_EXTENSION);
                $companyLogoBase64 = 'data:image/' . $extension . ';base64,' . base64_encode($logoData);
            }
        }

        // ۲. تولید PDF با mPDF
        $mpdf = new \Mpdf\Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4',
            'default_font' => 'vazir'
        ]);

        $html = view('livewire.admin.purchase.pdf-print', [
            'request' => $request,
            'companyLogo' => $companyLogoBase64, // ارسال داده‌های Base64 به جای URL
            'approvals' => $this->prepareApprovalsData($request), // متدی که اطلاعات امضاها را آماده می‌کند
        ])->render();

        $mpdf->WriteHTML($html);
        return response()->streamDownload(fn() => print($mpdf->Output('', 'S')), "PR-{$request->id}.pdf");
    }

//    public function downloadPdf($requestId): StreamedResponse
//    {
//        $request = PurchaseRequest::with(['project', 'user.company', 'items', 'approvals.user', 'approvals.role'])->findOrFail($requestId);
//
//        $companyLogoBase64 = null;
//        if ($request->user->company && $request->user->company->image) {
//            $logoPath = storage_path('app/public/company/logo/' . $request->user->company->image);
//            if (File::exists($logoPath)) {
//                $companyLogoBase64 = 'data:image/' . pathinfo($logoPath, PATHINFO_EXTENSION) . ';base64,' . base64_encode(File::get($logoPath));
//            }
//        }
//
//        $mpdf = new Mpdf([
//            'mode' => 'utf-8', 'format' => 'A4',
//            'fontDir' => [public_path('panel/dist/fonts/')],
//            'fontdata' => ['vazir' => ['R' => 'Vazir.ttf', 'B' => 'Vazir-Bold.ttf', 'useOTL' => 0xFF]],
//            'default_font' => 'vazir',
//        ]);
//        $mpdf->SetDirectionality('rtl');
//
//        $html = view('livewire.admin.purchase.pdf-print', [
//            'request' => $request,
//            'companyLogo' => $companyLogoBase64,
//            'approvals' => $request->approvals->sortBy('priority')->map(fn($app) => [
//                'role' => $app->role->name, 'name' => $app->user->name,
//                'signature' => $app->is_approved && $app->user->sign ? 'data:image/png;base64,' . base64_encode(File::get(storage_path('app/public/' . $app->user->sign))) : null,
//                'date' => $app->approved_at ? jdate($app->approved_at)->format('Y/m/d') : null
//            ]),
//        ])->render();
//
//        $mpdf->WriteHTML($html);
//        return response()->streamDownload(fn() => print($mpdf->Output('', 'S')), "PR-{$request->id}.pdf");
//    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.purchase.project-purchase-requests');
    }
}
