<?php

namespace App\Livewire\Admin\Purchase;

use Livewire\Attributes\Layout;
use Livewire\Component;
use App\Models\PurchaseRequest;
use App\Models\Project;
use App\Models\PurchaseRequestApproval;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Hekmatinasser\Verta\Verta;

class PurchaseDashboard extends Component
{
    public $chartRange = 30;

    public function updatedChartRange()
    {
        $this->dispatch('refreshCharts');
    }

    protected function getEfficiencyMetrics()
    {
        $avg = PurchaseRequestApproval::whereNotNull('approved_at')
            ->selectRaw('AVG(TIMESTAMPDIFF(HOUR, created_at, approved_at)) as avg_hours')
            ->value('avg_hours') ?? 0;

        $rejectionRate = PurchaseRequest::count() > 0
            ? (PurchaseRequest::where('status', 'rejected')->count() / PurchaseRequest::count()) * 100
            : 0;

        return [
            'days' => round($avg / 24, 1),
            'status' => $avg > 72 ? 'بحرانی' : ($avg > 48 ? 'کند' : 'ایده‌آل'),
            'color' => $avg > 72 ? 'danger' : ($avg > 48 ? 'warning' : 'success'),
            'rejection_rate' => round($rejectionRate, 1)
        ];
    }

    #[Layout('admin.master')]
    public function render()
    {
        // ۱. تحلیل مالی عمیق
        $totalAmount = DB::table('purchase_request_items')->sum(DB::raw('quantity * price')) ?? 0;
        $pendingAmount = DB::table('purchase_request_items')
            ->whereIn('purchase_request_id', PurchaseRequest::where('status', 'pending')->pluck('id'))
            ->sum(DB::raw('quantity * price')) ?? 0;

        // ۲. سلامت بودجه و پروژه‌ها
        $allProjects = Project::with(['mainContract.addenda', 'purchaseRequests.items'])->get();
        $totalBudget = $allProjects->sum('budget');
        $budgetUtilization = $totalBudget > 0 ? round(($totalAmount / $totalBudget) * 100, 1) : 0;

        $criticalProjects = $allProjects->map(function ($project) {
            $project->spent = $project->purchaseRequests->sum(fn($pr) => $pr->items->sum(fn($i) => $i->quantity * $i->price));
            $project->utilization = $project->budget > 0 ? round(($project->spent / $project->budget) * 100, 1) : 0;
            return $project;
        })->sortByDesc('utilization')->take(5);

        // ۳. متریک‌های فرآیندی
        $efficiency = $this->getEfficiencyMetrics();

        // ۴. نمودار روند (Trends)
        $trendLabels = []; $trendData = [];
        for ($i = $this->chartRange - 1; $i >= 0; $i--) {
            $date = now()->subDays($i)->startOfDay();
            $trendLabels[] = Verta::instance($date)->format('m/d');
            $trendData[] = PurchaseRequest::whereDate('created_at', $date)->count();
        }

        return view('livewire.admin.purchase.purchase-dashboard', [
            'totalRequests' => PurchaseRequest::count(),
            'totalAmount' => $totalAmount,
            'pendingAmount' => $pendingAmount,
            'statusData' => [
                'pending' => PurchaseRequest::where('status', 'pending')->count(),
                'approved' => PurchaseRequest::where('status', 'approved')->count(),
                'rejected' => PurchaseRequest::where('status', 'rejected')->count(),
            ],
            'efficiency' => $efficiency,
            'budgetUtilization' => $budgetUtilization,
            'availableBudget' => $totalBudget - $totalAmount,
            'criticalProjects' => $criticalProjects,
            'myPendingApprovals' => PurchaseRequestApproval::with(['request.project', 'request.user'])
                ->where('user_id', Auth::id())->whereNull('is_approved')->whereHas('request')->latest()->take(5)->get(),
            'latestRequests' => PurchaseRequest::with(['project', 'user', 'approvals.user'])->latest()->take(10)->get(),
            'trendLabels' => $trendLabels,
            'trendData' => $trendData,
        ]);
    }
}
