<?php

namespace App\Livewire\Admin\Purchase;

use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;
use App\Models\PurchaseRequestApproval as ApprovalModel;
use App\Models\Project;

class PurchaseRequestApproval extends Component
{
    use WithPagination;
    protected $paginationTheme = 'bootstrap';

    public $search = '';
    public $projectFilter = '';
    public $tab = 'pending'; // 'pending' | 'history'
    public $comment = []; // کلید: approval_id

    // ریست صفحه هنگام تغییر فیلترها
    public function updatingSearch() { $this->resetPage(); }
    public function updatingProjectFilter() { $this->resetPage(); }
    public function updatingTab() { $this->resetPage(); }

    #[Layout('admin.master')]
    public function render()
    {
        $user = Auth::user();

        // آمار خلاصه
        $pendingCount = ApprovalModel::where('user_id', $user->id)->whereNull('is_approved')->count();
        $approvedCount = ApprovalModel::where('user_id', $user->id)->where('is_approved', true)->count();
        $rejectedCount = ApprovalModel::where('user_id', $user->id)->where('is_approved', false)->count();

        // تب فعال
        $approvals = ApprovalModel::with([
            'user',
            'request.project',
            'request.items',
            'request.user',
            'request.approvals.user',
            'request.approvals.role'
        ])
            ->where('user_id', $user->id)
            ->when($this->tab === 'pending', function ($query) {
                $query->whereNull('is_approved');
            }, function ($query) {
                $query->whereNotNull('is_approved');
            })
            ->when($this->search, function ($query) {
                $query->whereHas('request', function ($q) {
                    $q->where('description', 'like', '%' . $this->search . '%')
                        ->orWhereHas('project', function ($qq) {
                            $qq->where('name', 'like', '%' . $this->search . '%');
                        })
                        ->orWhereHas('user', function ($qq) {
                            $qq->where('name', 'like', '%' . $this->search . '%');
                        });
                });
            })
            ->when($this->projectFilter, function ($query) {
                $query->whereHas('request', function ($q) {
                    $q->where('project_id', $this->projectFilter);
                });
            })
            ->orderByDesc('id')
            ->paginate(10);

        $projects = Project::all();

        return view('livewire.admin.purchase.purchase-request-approval', [
            'approvals' => $approvals,
            'projects' => $projects,
            'pendingCount' => $pendingCount,
            'approvedCount' => $approvedCount,
            'rejectedCount' => $rejectedCount,
            'tab' => $this->tab,
            'search' => $this->search,
            'projectFilter' => $this->projectFilter,
        ]);
    }

    public function approve($approvalId)
    {
        $approval = ApprovalModel::findOrFail($approvalId);
        if ($approval->is_approved !== null) return;
        $approval->update([
            'is_approved' => true,
            'approved_at' => now(),
            'comment' => $this->comment[$approvalId] ?? null,
        ]);
        session()->flash('message', 'درخواست با موفقیت تأیید شد.');
        unset($this->comment[$approvalId]);
    }

    public function reject($approvalId)
    {
        $approval = ApprovalModel::findOrFail($approvalId);
        if ($approval->is_approved !== null) return;
        $approval->update([
            'is_approved' => false,
            'approved_at' => now(),
            'comment' => $this->comment[$approvalId] ?? null,
        ]);
        session()->flash('message', 'درخواست با موفقیت رد شد.');
        unset($this->comment[$approvalId]);
    }

    public function return($approvalId)
    {
        $approval = ApprovalModel::findOrFail($approvalId);
        $approval->update([
            'is_approved' => null,
            'comment' => $this->comment[$approvalId] ?? null,
        ]);
        session()->flash('message', 'درخواست برای اصلاح برگشت داده شد.');
        unset($this->comment[$approvalId]);
    }
}
