<?php

namespace App\Livewire\Admin\Purchase;

use App\Models\Item;
use App\Models\Project;
use App\Models\PurchaseRequest;
use App\Models\ApprovalRule;
use App\Models\PurchaseRequestApproval;
use App\Models\User;
use App\Notifications\PurchaseRequestCreated;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB; // اضافه شد
use Illuminate\Support\Facades\Notification;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate;
use Morilog\Jalali\Jalalian;

class PurchaseRequestManagement extends Component
{
    use WithPagination;

    public $itemSearch = '';
    public $projectSearch = '';

    #[Validate('required|exists:projects,id')]
    public $project_id;

    #[Validate('required|string|max:500')]
    public $description;

    #[Validate('required|in:material,service,asset')]
    public $purchase_type = 'material';

    #[Validate('required|integer')]
    public $cost_center_id;

    #[Validate('required|string|min:10')]
    public $required_by;

    public $search = '';
    public $statusFilter = '';
    public $projectFilter = '';
    protected $paginationTheme = 'bootstrap';

    public function mount()
    {
        $urlProjectId = request()->query('project_id');
        if ($urlProjectId) {
            $project = Project::find($urlProjectId);
            if ($project) {
                $this->project_id = $project->id;
                $this->projectSearch = $project->name;
            }
        }
        $this->required_by = Jalalian::forge('today')->format('Y/m/d');
    }

    #[Computed]
    public function availableItemsList()
    {
        return Item::when($this->itemSearch, function ($query) {
            $query->where('name', 'like', '%' . $this->itemSearch . '%')
                ->orWhere('code', 'like', '%' . $this->itemSearch . '%');
        })->limit(20)->get();
    }

    #[Computed]
    public function projectsList()
    {
        return Project::when($this->projectSearch, function ($query) {
            $query->where('name', 'like', '%' . $this->projectSearch . '%')
                ->orWhere('code', 'like', '%' . $this->projectSearch . '%');
        })->limit(10)->get();
    }

    #[Computed]
    public function myPurchaseRequests()
    {
        // بهینه‌سازی لود داده‌ها برای آکاردئون
        return PurchaseRequest::with(['project', 'items.item', 'approvals.user', 'approvals.role'])
            ->where('user_id', Auth::id())
            ->when($this->search, fn($q) => $q->where('description', 'like', '%' . $this->search . '%'))
            ->when($this->statusFilter, fn($q) => $q->where('status', $this->statusFilter))
            ->when($this->projectFilter, fn($q) => $q->where('project_id', $this->projectFilter))
            ->orderBy('created_at', 'desc')
            ->paginate(10);
    }

    #[On('save-final-request')]
    public function savePurchaseRequest(array $finalItems)
    {
        $this->validate();

        if (empty($finalItems)) {
            $this->dispatch('alert', type: 'error', message: 'لیست کالاها نمی‌تواند خالی باشد.');
            return;
        }

        // استفاده از Transaction برای امنیت داده‌ها
        DB::beginTransaction();

        try {
            $requiredByCarbon = Jalalian::fromFormat('Y/m/d', $this->required_by)->toCarbon();

            $purchaseRequest = PurchaseRequest::create([
                'user_id' => Auth::id(),
                'project_id' => $this->project_id,
                'description' => $this->description,
                'purchase_type' => $this->purchase_type,
                'cost_center_id' => $this->cost_center_id,
                'required_by' => $requiredByCarbon,
                'status' => 'pending',
            ]);

            foreach ($finalItems as $item) {
                $purchaseRequest->items()->create([
                    'item_id' => $item['item_id'] ?? null,
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'description' => $item['is_service_item'] ? $item['name'] : null,
                ]);
            }

            $totalAmount = collect($finalItems)->sum(fn($i) => $i['quantity'] * $i['price']);
            $this->applyWorkflow($purchaseRequest, $totalAmount);

            DB::commit(); // تایید نهایی ذخیره‌سازی

            // ارسال نوتیفیکیشن بعد از Commit موفق
            $this->notifyApprovers($purchaseRequest);

            $this->resetFields();
            $this->dispatch('alert', type: 'success', message: 'درخواست PR-'.$purchaseRequest->id.' با موفقیت ثبت و به جریان افتاد.');

        } catch (\Exception $e) {
            DB::rollBack(); // لغو تغییرات در صورت بروز خطا
            $this->dispatch('alert', type: 'error', message: 'خطای سیستمی: ' . $e->getMessage());
        }
    }

    private function applyWorkflow($request, $totalAmount)
    {
        $rules = ApprovalRule::with('approvers')
            ->where(function($q) {
                $q->where('project_id', $this->project_id)->orWhere('is_default', 1);
            })
            ->where('is_active', 1)
            ->orderBy('priority', 'asc')
            ->get();

        foreach ($rules as $rule) {
            foreach ($rule->approvers as $approver) {
                PurchaseRequestApproval::create([
                    'purchase_request_id' => $request->id,
                    'user_id' => $approver->user_id,
                    'role_id' => $approver->role_id,
                    'priority' => $approver->priority,
                    'is_approved' => null,
                ]);
            }
        }
    }

    private function notifyApprovers($request)
    {
        // اصلاح باگ: چک کردن وجود تاییدیه قبل از گرفتن اولویت
        $firstStepPriority = $request->approvals()->min('priority');

        if ($firstStepPriority !== null) {
            $approverIds = $request->approvals()
                ->where('priority', $firstStepPriority)
                ->pluck('user_id');

            $usersToNotify = User::whereIn('id', $approverIds)->get();

            if ($usersToNotify->isNotEmpty()) {
                Notification::send($usersToNotify, new PurchaseRequestCreated($request));
            }
        }
    }

    public function deleteRequest($id)
    {
        $request = PurchaseRequest::findOrFail($id);

        if ($request->user_id !== Auth::id()) {
            $this->dispatch('alert', type: 'error', message: 'عدم دسترسی: این درخواست متعلق به شما نیست.');
            return;
        }

        if ($request->status !== 'pending') {
            $this->dispatch('alert', type: 'warning', message: 'فقط درخواست‌های در انتظار قابل حذف هستند.');
            return;
        }

        $request->delete();
        $this->dispatch('alert', type: 'success', message: 'درخواست با موفقیت ابطال شد.');
    }

    public function resetFields()
    {
        $this->reset(['project_id', 'description', 'projectSearch', 'itemSearch']);
        $this->required_by = Jalalian::forge('today')->format('Y/m/d');
        $this->dispatch('request-reset-success');
    }

    #[Layout('admin.master')]
    public function render()
    {
        $quickStats = [
            'total' => PurchaseRequest::where('user_id', Auth::id())->count(),
            'pending' => PurchaseRequest::where('user_id', Auth::id())->where('status', 'pending')->count(),
            'approved' => PurchaseRequest::where('user_id', Auth::id())->where('status', 'approved')->count(),
        ];

        return view('livewire.admin.purchase.purchase-request-management', [
            'costCenters' => [1 => 'خدمات مهندسی', 2 => 'تجهیزات تخصصی', 3 => 'هزینه‌های اداری'],
            'purchaseTypes' => ['material' => 'کالا / مواد', 'service' => 'خدمات فنی', 'asset' => 'دارایی ثابت'],
            'quickStats' => $quickStats
        ]);
    }
}
