<?php

namespace App\Livewire\Admin\Roles;

use App\Models\Role;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;
use Illuminate\Validation\Rule as ValidationRule; // 💡 استفاده از Rule لاراول برای Unique

#[Layout('admin.master')]
class AddRoles extends Component
{
    // 🛑 FIX: استفاده از ?string برای پذیرش null از دیتابیس
    public ?string $name = '';
    public ?string $label = '';

    public bool $isProjectRole = false;
    public ?int $roleId = null;
    public bool $isEditMode = false;

    // قانون اعتبارسنجی پایه - نام نمایشی نیازی به Unique بودن ندارد.
    protected $rules = [
        'label' => 'required|min:2|string|max:100',
        'isProjectRole' => 'boolean',
    ];

    protected array $messages = [
        'name.required' => 'نام فنی نقش الزامی است.',
        'name.min' => 'نام فنی باید حداقل ۲ کاراکتر باشد.',
        'label.required' => 'نام نمایشی (فارسی) نقش الزامی است.',
    ];

    // 🚨 گارد امنیتی: کل کامپوننت فقط برای ادمین اصلی قابل دسترسی است.
    public function boot(): void
    {
        if (Gate::denies('is-admin')) {
            abort(403, 'شما اجازه مدیریت نقش‌های سیستم را ندارید.');
        }
    }

    #[On('role-updated')]
    #[On('role-deleted')]
    #[On('role-created')]
    public function showMessage(string $message): void
    {
        session()->flash('success', $message);
    }

    #[Layout('admin.master')]
    public function render()
    {
        $roles = Role::select('id', 'name', 'label', 'is_project_role')->orderBy('id', 'desc')->get();
        return view('livewire.admin.roles.add-roles', compact('roles'));
    }

    // ثبت نقش جدید
    public function saveRole(): void
    {
        // 🛑 ساختن قوانین نهایی شامل Unique بودن نام فنی
        $rules = array_merge($this->rules, [
            'name' => ['required', 'min:2', 'string', 'max:50', 'unique:roles,name'],
        ]);
        $this->validate($rules);

        Role::create([
            'name' => $this->name,
            'label' => $this->label,
            'is_project_role' => $this->isProjectRole,
        ]);

        $this->resetInput();
        $this->dispatch('role-created', message: 'نقش با موفقیت ایجاد گردید.');
    }

    // بارگذاری نقش برای ویرایش
    public function editRole(Role $role): void
    {
        $this->roleId = $role->id;
        $this->name = $role->name;
        $this->label = $role->label;
        $this->isProjectRole = $role->is_project_role;
        $this->isEditMode = true;
    }

    // به‌روزرسانی نقش (شامل FIX Unique)
    public function updateRole(): void
    {
        if (!$this->isEditMode || !$this->roleId) {
            $this->dispatch('error', message: 'خطا: در وضعیت ویرایش نیستید.');
            return;
        }

        // 🛑 FIX: ساختن قوانین نهایی با نادیده گرفتن ID فعلی
        $rules = array_merge($this->rules, [
            'name' => ['required', 'min:2', 'string', 'max:50', ValidationRule::unique('roles', 'name')->ignore($this->roleId)],
        ]);
        $this->validate($rules);

        $role = Role::findOrFail($this->roleId);
        $role->update([
            'name' => $this->name,
            'label' => $this->label,
            'is_project_role' => $this->isProjectRole,
        ]);

        $this->resetInput();
        $this->dispatch('role-updated', message: 'نقش با موفقیت به‌روزرسانی شد.');
    }

    public function deleteRole(Role $role): void
    {
        $role->delete();
        $this->dispatch('role-deleted', message: 'نقش با موفقیت حذف شد.');
    }

    public function resetInput(): void
    {
        $this->name = '';
        $this->label = '';
        $this->isProjectRole = false;
        $this->reset(['roleId', 'isEditMode']);
    }
}
