<?php

namespace App\Livewire\Admin\Roles;

use App\Models\Role;
use App\Models\Permission;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule; // برای استفاده در ساختار کلاس

#[Layout('admin.master')]
class RolePermissionManager extends Component
{
    // لیست تمام نقش‌ها و مجوزها
    public $roles;
    public $permissions;

    // آرایه‌ای برای ذخیره وضعیت تیک‌ها (RoleID => [PermissionID => true/false])
    public array $assignedPermissions = [];

    // 💡 متغیر برای مدیریت وضعیت لودینگ دستی (هوشمندانه تر)
    public bool $isProcessing = false;

    public function boot(): void
    {
        // 🚨 گارد امنیتی: کل کامپوننت فقط برای ادمین اصلی قابل دسترسی است.
        if (Gate::denies('is-admin')) {
            abort(403, 'شما اجازه مدیریت نقش‌ها و مجوزها را ندارید.');
        }
    }

    public function mount()
    {
        $this->loadData();
    }

    public function loadData(): void
    {
        // 1. واکشی نقش‌ها: حذف نقش 'admin' از لیست قابل ویرایش توسط کاربران
        $this->roles = Role::where('name', '!=', 'admin')
            ->with('permissions:id') // واکشی ID مجوزها برای ساخت ماتریس
            ->orderBy('label', 'asc')
            ->get();

        // 2. واکشی مجوزها: (انتخاب label و name برای نمایش)
        $this->permissions = Permission::orderBy('group')->orderBy('label')->get();

        // 3. ایجاد ماتریس اولیه وضعیت تیک‌ها
        $this->assignedPermissions = []; // ریست اولیه
        $this->roles->each(function ($role) {
            $permissionIds = $role->permissions->pluck('id')->toArray();

            $this->permissions->each(function ($permission) use ($role, $permissionIds) {
                $this->assignedPermissions[$role->id][$permission->id] = in_array($permission->id, $permissionIds);
            });
        });
    }

    /**
     * @param int $roleId
     * @param int $permissionId
     * @param bool $status
     * منطق اصلی برای تیک زدن/برداشتن تیک
     */
    public function togglePermission(int $roleId, int $permissionId, bool $status): void
    {
        // 💡 فعال سازی لودینگ
        $this->isProcessing = true;

        $role = $this->roles->firstWhere('id', $roleId);

        if (!$role || !Permission::find($permissionId)) {
            session()->flash('error', 'عملیات نامعتبر یا داده‌های ارسالی ناقص هستند.');
            $this->loadData();
            $this->isProcessing = false;
            return;
        }

        try {
            if ($status) {
                // ✅ استفاده از متد attach()
                $role->permissions()->attach($permissionId);
            } else {
                // ✅ استفاده از متد detach()
                $role->permissions()->detach($permissionId);
            }

            $this->assignedPermissions[$roleId][$permissionId] = $status;
            session()->flash('success', 'دسترسی با موفقیت به‌روزرسانی شد.');
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در ثبت تغییرات. لطفاً مجدداً تلاش کنید.');
            $this->loadData();
        }

        $this->isProcessing = false; // غیرفعال سازی لودینگ
    }

    // 💡 متد برای رفرش کردن کامپوننت پس از عملیات (اگر از رویدادها استفاده شود)
    #[On('permissions-updated')]
    public function refreshComponent(): void
    {
        $this->loadData();
    }

    public function render()
    {
        return view('livewire.admin.roles.role-permission-manager');
    }
}
