<?php

namespace App\Livewire\Admin\Sections;

use App\Models\{Section, ProjectFile, Folder, FileActivity};
use Livewire\{Component, WithFileUploads, Attributes\Layout, Attributes\Url, Attributes\Computed};
use Illuminate\Support\Facades\{Storage, File, DB, Auth, Log};
use Illuminate\Support\Str;
use Morilog\Jalali\Jalalian;
use ZipArchive;

class SectionFileUpload extends Component
{
    use WithFileUploads;

    public Section $section;

    // --- STATES ---
    public array $files = [];
    public array $fileNames = [];
    public array $isPrivate = [];
    public array $tags = [];
    public array $descriptions = [];
    public int $uploadProgress = 0;

    #[Url(as: 'q', history: true)] public string $search = '';
    #[Url(as: 'folder')] public $currentFolderId = null;
    #[Url] public string $viewMode = 'grid'; // grid | table

    public string $sortField = 'created_at';
    public string $sortDirection = 'desc';

    // --- SELECTION & BATCH ---
    public array $selectedFiles = [];
    public array $selectedFolders = [];
    public $targetFolderId = null;

    // --- MODALS DATA ---
    public string $newFolderName = '';
    public $editingFileId = null;
    public string $editedFileName = '';
    public string $editedDescription = '';
    public string $editedTags = '';
    public ?ProjectFile $historyFile = null;
    public array $fileHistory = [];

    protected function rules(): array
    {
        return [
            'files.*' => 'required|file|max:102400|mimes:jpg,jpeg,png,pdf,docx,zip,rar,xls,xlsx,ppt,pptx,dwg,psd',
            'newFolderName' => 'required|string|min:1|max:100',
            'editedFileName' => 'required|string|min:2|max:255',
        ];
    }

    public function mount(Section $section)
    {
        $this->section = $section->load('project');
    }

    // --- COMPUTED PROPERTIES ---

    #[Computed]
    public function folders()
    {
        return Folder::where('section_id', $this->section->id)
            ->where('parent_id', $this->currentFolderId)
            ->when($this->search, fn($q) => $q->where('name', 'like', "%{$this->search}%"))
            ->withCount('files')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function existingFiles()
    {
        return ProjectFile::where('section_id', $this->section->id)
            ->where('folder_id', $this->currentFolderId)
            ->with(['user:id,name'])
            ->when($this->search, function($q) {
                $q->where(function($query) {
                    $query->where('name', 'like', "%{$this->search}%")
                        ->orWhere('tags', 'like', "%{$this->search}%")
                        ->orWhere('description', 'like', "%{$this->search}%");
                });
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->get();
    }

    #[Computed]
    public function breadcrumbs()
    {
        $crumbs = collect();
        $folder = Folder::find($this->currentFolderId);
        while ($folder) {
            $crumbs->prepend($folder);
            $folder = Folder::find($folder->parent_id);
        }
        return $crumbs;
    }

    // --- CORE OPERATIONS ---

    public function updatedFiles()
    {
        $this->validateOnly('files.*');
        foreach ($this->files as $index => $file) {
            $this->fileNames[$index] = $this->fileNames[$index] ?? pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        }
    }
    /**
     * بارگذاری اطلاعات فایل برای نمایش در مودال ویرایش
     */
    public function editFile($id)
    {
        $file = ProjectFile::findOrFail($id);

        $this->editingFileId = $id;
        $this->editedFileName = $file->name;
        $this->editedDescription = $file->description ?? '';
        // تبدیل تگ‌ها از آرایه به رشته برای نمایش در اینپوت
        $this->editedTags = is_array($file->tags) ? implode(', ', $file->tags) : '';

        // فراخوانی مودال در سمت کلاینت
        $this->dispatch('show-edit-modal');
    }

    /**
     * ذخیره تغییرات اعمال شده روی فایل
     */
    public function updateFile()
    {
        // اعتبارسنجی فیلدها
        $this->validate([
            'editedFileName' => 'required|string|min:2|max:255',
            'editedDescription' => 'nullable|string|max:1000',
        ]);

        $file = ProjectFile::findOrFail($this->editingFileId);

        // ثبت مقادیر قدیمی برای لاگ تاریخچه (اختیاری)
        $oldData = $file->only(['name', 'description', 'tags']);

        $file->update([
            'name' => $this->editedFileName,
            'description' => $this->editedDescription,
            // تبدیل رشته تگ‌ها به آرایه برای ذخیره در دیتابیس
            'tags' => array_map('trim', explode(',', $this->editedTags)),
        ]);

        // ثبت فعالیت در تاریخچه
        $this->logActivity($file->id, 'METADATA_EDIT', 'به‌روزرسانی مشخصات فنی سند');

        // بستن مودال و نمایش پیام موفقیت
        $this->dispatch('hide-edit-modal');
        $this->dispatch('alert', ['type' => 'success', 'message' => 'مشخصات سند با موفقیت بروزرسانی شد.']);
    }


    public function uploadFiles()
    {
        $this->validate(['files.*' => 'required|file']);

        DB::transaction(function () {
            $directory = "projects/{$this->section->project_id}/sections/{$this->section->id}";
            foreach ($this->files as $index => $file) {
                $path = $file->storeAs($directory, Str::slug($this->fileNames[$index]).'_'.uniqid().'.'.$file->extension(), 'public');

                $newFile = ProjectFile::create([
                    'name' => $this->fileNames[$index],
                    'path' => $path,
                    'section_id' => $this->section->id,
                    'project_id' => $this->section->project_id,
                    'folder_id' => $this->currentFolderId,
                    'user_id' => Auth::id(),
                    'size' => $file->getSize(),
                    'type' => $file->getMimeType(),
                    'is_private' => $this->isPrivate[$index] ?? false,
                    'tags' => array_filter(explode(',', $this->tags[$index] ?? '')),
                    'description' => $this->descriptions[$index] ?? '',
                    'version' => 1,
                ]);

                $this->logActivity($newFile->id, 'UPLOAD', "فایل '{$newFile->name}' آرشیو شد.");
            }
        });

        $this->reset(['files', 'fileNames', 'tags', 'descriptions']);
        $this->dispatch('alert', ['type' => 'success', 'message' => 'مستندات با موفقیت ثبت شدند.']);
    }

    // --- FOLDER LOGIC ---

    public function createNewFolder()
    {
        $this->validate(['newFolderName' => 'required|max:100']);

        // جلوگیری از نام تکراری در مسیر فعلی
        $exists = Folder::where('parent_id', $this->currentFolderId)
            ->where('section_id', $this->section->id)
            ->where('name', $this->newFolderName)
            ->exists();

        if ($exists) {
            $this->dispatch('alert', ['type' => 'error', 'message' => 'پوشه‌ای با این نام در این مسیر وجود دارد.']);
            return;
        }

        Folder::create([
            'name' => $this->newFolderName,
            'parent_id' => $this->currentFolderId,
            'section_id' => $this->section->id,
            'user_id' => Auth::id()
        ]);

        $this->newFolderName = '';
        $this->dispatch('alert', ['type' => 'success', 'message' => 'پوشه ساخته شد.']);
    }

    public function enterFolder($id) { $this->currentFolderId = $id; $this->resetSelection(); }
    public function backToParent() { $this->currentFolderId = Folder::find($this->currentFolderId)?->parent_id; }
    public function backToRoot() { $this->currentFolderId = null; }

    // --- ADVANCED ZIP DOWNLOAD (Tree Structure) ---

    public function downloadSelectedAsZip()
    {
        if (empty($this->selectedFiles) && empty($this->selectedFolders)) return;

        $zipName = 'DCC_Archive_'.now()->format('Ymd_His').'.zip';
        $zipPath = storage_path('app/public/temp/'.$zipName);
        if(!File::isDirectory(storage_path('app/public/temp'))) File::makeDirectory(storage_path('app/public/temp'), 0755, true);

        $zip = new ZipArchive;
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {

            // 1. افزودن فایل‌های مستقیماً انتخاب شده
            foreach ($this->selectedFiles as $fileId) {
                $file = ProjectFile::find($fileId);
                if ($file && Storage::disk('public')->exists($file->path)) {
                    $zip->addFile(Storage::disk('public')->path($file->path), $file->name . '.' . File::extension($file->path));
                }
            }

            // 2. افزودن پوشه‌های انتخاب شده و محتویات آن‌ها به صورت بازگشتی
            foreach ($this->selectedFolders as $folderId) {
                $folder = Folder::find($folderId);
                if ($folder) {
                    $this->addFolderToZip($folder, $zip, $folder->name);
                }
            }

            $zip->close();
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }

    private function addFolderToZip($folder, $zip, $relativeName)
    {
        $zip->addEmptyDir($relativeName);

        // افزودن فایل‌های این پوشه
        $files = ProjectFile::where('folder_id', $folder->id)->get();
        foreach ($files as $file) {
            if (Storage::disk('public')->exists($file->path)) {
                $zip->addFile(Storage::disk('public')->path($file->path), $relativeName . '/' . $file->name . '.' . File::extension($file->path));
            }
        }

        // فراخوانی بازگشتی برای زیرپوشه‌ها
        $subFolders = Folder::where('parent_id', $folder->id)->get();
        foreach ($subFolders as $sub) {
            $this->addFolderToZip($sub, $zip, $relativeName . '/' . $sub->name);
        }
    }

    // --- BATCH OPERATIONS ---

//    public function moveItems()
//    {
//        $this->validate(['targetFolderId' => 'required|exists:folders,id']);
//
//        ProjectFile::whereIn('id', $this->selectedFiles)->update(['folder_id' => $this->targetFolderId]);
//        Folder::whereIn('id', $this->selectedFolders)->update(['parent_id' => $this->targetFolderId]);
//
//        $this->resetSelection();
//        $this->dispatch('alert', ['type' => 'success', 'message' => 'جابجایی با موفقیت انجام شد.']);
//    }
    public function moveItems()
    {
        $this->validate(['targetFolderId' => 'required']);

        DB::transaction(function () {
            if (!empty($this->selectedFiles)) {
                ProjectFile::whereIn('id', $this->selectedFiles)->update([
                    'folder_id' => $this->targetFolderId == 'root' ? null : $this->targetFolderId
                ]);
            }
            if (!empty($this->selectedFolders)) {
                // جلوگیری از جابجایی پوشه به داخل خودش
                Folder::whereIn('id', $this->selectedFolders)
                    ->where('id', '!=', $this->targetFolderId)
                    ->update([
                        'parent_id' => $this->targetFolderId == 'root' ? null : $this->targetFolderId
                    ]);
            }
        });

        $this->resetSelection();
        $this->dispatch('alert', ['type' => 'success', 'message' => 'جابجایی با موفقیت انجام شد.']);
    }


    public function deleteSelectedFiles()
    {
        if (!Auth::user()->isAdmin()) return;

        DB::transaction(function () {
            // حذف فایل‌ها
            $files = ProjectFile::whereIn('id', $this->selectedFiles)->get();
            foreach ($files as $file) {
                Storage::disk('public')->delete($file->path);
                $file->delete();
            }

            // حذف پوشه‌ها و محتویات (Recursive)
            foreach ($this->selectedFolders as $folderId) {
                $this->deleteFolderRecursive($folderId);
            }
        });

        $this->resetSelection();
        $this->dispatch('alert', ['type' => 'warning', 'message' => 'موارد انتخابی پاکسازی شدند.']);
    }

    private function deleteFolderRecursive($folderId)
    {
        $files = ProjectFile::where('folder_id', $folderId)->get();
        foreach ($files as $file) {
            Storage::disk('public')->delete($file->path);
            $file->delete();
        }

        $subFolders = Folder::where('parent_id', $folderId)->get();
        foreach ($subFolders as $sub) {
            $this->deleteFolderRecursive($sub->id);
        }

        Folder::find($folderId)?->delete();
    }

    // --- HELPERS ---

    public function selectAll()
    {
        $fileIds = $this->existingFiles->pluck('id')->map(fn($id) => (string)$id)->toArray();
        $folderIds = $this->folders->pluck('id')->map(fn($id) => (string)$id)->toArray();

        if (count($this->selectedFiles) === count($fileIds) && count($this->selectedFolders) === count($folderIds)) {
            $this->selectedFiles = [];
            $this->selectedFolders = [];
        } else {
            $this->selectedFiles = $fileIds;
            $this->selectedFolders = $folderIds;
        }
    }

    public function resetSelection() { $this->reset(['selectedFiles', 'selectedFolders', 'targetFolderId']); }

    private function logActivity($fileId, $type, $details) {
        FileActivity::create(['project_file_id' => $fileId, 'user_id' => Auth::id(), 'action_type' => $type, 'details' => $details]);
    }

//    public function getFileIcon($mime): string
//    {
//        return match (true) {
//            Str::contains($mime, 'pdf') => 'ti-file-type-pdf text-danger',
//            Str::contains($mime, ['zip', 'rar']) => 'ti-file-zip text-warning',
//            Str::contains($mime, ['excel', 'sheet']) => 'ti-file-type-xls text-success',
//            Str::contains($mime, 'word') => 'ti-file-type-doc text-primary',
//            Str::contains($mime, 'image') => 'ti-photo text-info',
//            default => 'ti-file text-secondary',
//        };
//    }
    public function getFileIcon($mime): string
    {
        return match (true) {
            Str::contains($mime, 'pdf') => 'ti-file-type-pdf',
            Str::contains($mime, ['zip', 'rar', '7z']) => 'ti-file-zip',
            Str::contains($mime, ['excel', 'sheet', 'xls', 'xlsx', 'csv']) => 'ti-file-type-xls',
            Str::contains($mime, ['word', 'doc', 'docx']) => 'ti-file-type-doc',
            Str::contains($mime, ['ppt', 'pptx']) => 'ti-file-type-ppt',
            Str::contains($mime, 'image') => 'ti-photo',
            Str::contains($mime, 'text') => 'ti-file-text',
            Str::contains($mime, ['dwg', 'dxf']) => 'ti-vector-bezier',
            default => 'ti-file-description',
        };
    }
    public function formatSize($bytes): string
    {
        if ($bytes >= 1073741824) return round($bytes / 1073741824, 2) . ' GB';
        if ($bytes >= 1048576) return round($bytes / 1048576, 2) . ' MB';
        return round($bytes / 1024, 1) . ' KB';
    }

    /**
     * بازیابی تاریخچه فعالیت‌های یک فایل خاص
     */
    public function showHistory($id)
    {
        // واکشی فایل به همراه فعالیت‌های مرتبط و کاربران
        $this->historyFile = ProjectFile::with(['activities.user', 'user'])->findOrFail($id);

        // تبدیل فعالیت‌ها به آرایه‌ای خوانا برای نمایش در تایم‌لاین
        $this->fileHistory = $this->historyFile->activities->map(fn($a) => [
            'action' => $this->parseActionType($a->action_type),
            'user' => $a->user->name ?? 'سیستم',
            'date' => Jalalian::fromDateTime($a->created_at)->format('Y/m/d H:i'),
            'details' => $a->details
        ])->toArray();

        // ارسال سیگنال برای باز شدن مودال در فرانت‌اِند
        $this->dispatch('show-history-modal');
    }

    /**
     * تبدیل کدهای سیستمی به متون فارسی برای نمایش در تاریخچه
     */
    private function parseActionType($type)
    {
        return match($type) {
            'UPLOAD' => 'بارگذاری اولیه',
            'DOWNLOAD' => 'دریافت فایل (Download)',
            'METADATA_EDIT' => 'ویرایش شناسنامه سند',
            'MOVE' => 'جابجایی در پوشه‌ها',
            'DELETE' => 'حذف موقت',
            default => $type
        };
    }

    /**
     * ریست کردن فرم‌ها و متغیرهای موقت مودال
     */
    public function resetForms()
    {
        $this->reset([
            'editingFileId',
            'editedFileName',
            'editedDescription',
            'editedTags',
            'newFolderName',
            'targetFolderId',
            'historyFile',
            'fileHistory'
        ]);

        // پاکسازی خطاهای اعتبارسنجی
        $this->resetErrorBag();
    }

    // اضافه کردن متد محاسبه حجم پوشه به صورت داینامیک
    public function getFolderSize($folderId)
    {
        $size = ProjectFile::where('folder_id', $folderId)->sum('size');
        $subFolders = Folder::where('parent_id', $folderId)->get();
        foreach ($subFolders as $sub) {
            $size += $this->getFolderSize($sub->id);
        }
        return $this->formatSize($size);
    }



// متد جدید برای باز کردن پیش‌نمایش
    public function openPreview($fileId)
    {
        $file = ProjectFile::findOrFail($fileId);
        $url = Storage::disk('public')->url($file->path);
        $this->dispatch('open-preview-modal', ['url' => $url, 'type' => $file->type, 'name' => $file->name]);
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.sections.section-file-upload', [
            'allFolders' => Folder::where('section_id', $this->section->id)
                ->where('id', '!=', $this->currentFolderId)->get()
        ]);
    }
}
