<?php
// FILE: App/Livewire/Admin/Task/CommentList.php

namespace App\Livewire\Admin\Task;

use App\Models\TaskComment;
use App\Models\TaskCommentView;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class CommentList extends Component
{
    use WithFileUploads;

    public $taskId;
    public $body = '';
    public $replyTo = null;
    public $file;
    public $task; // 💡 آبجکت تسک برای چک کردن مسئول

    // 💡 NEW: برای مدیریت محتوای پاسخ‌ها (keyed by parent_id)
    public $replyBody = [];

    protected $listeners = ['setReply'];
    protected $rules = [
        'body' => 'required|string|min:2',
        'file' => 'nullable|file|max:10240', // آپلود تا 10 مگابایت
    ];

    public function mount($taskId)
    {
        $this->taskId = $taskId;
        // 💡 واکشی تسک برای استفاده در تابع کمکی renderComments
        $this->task = \App\Models\Task::select('id', 'assigned_to')->with('assignedUser:id,name')->find($taskId);
    }

    /**
     * ثبت کامنت جدید (با یا بدون parent_id)
     */
    public function addComment($parentId = null)
    {
        // 🛑 FIX: اگر parentId موجود باشد، از آرایه replyBody محتوا را بگیر
        $bodyContent = $parentId ? ($this->replyBody[$parentId] ?? null) : $this->body;

        if (empty($bodyContent) || strlen($bodyContent) < 2) {
            session()->flash('comment-alert', 'محتوای نظر نمی‌تواند خالی باشد.');
            return;
        }

        // اگر سطح اول است، body را اعتبارسنجی کن، در غیر این صورت صرف‌نظر کن.
        if (!$parentId) {
            $this->validate(['body' => 'required|string|min:2', 'file' => $this->rules['file']]);
        }

        $path = $this->file ? $this->file->store('comment_files', 'public') : null;

        \App\Models\TaskComment::create([
            'task_id'   => $this->taskId,
            'user_id'   => auth()->id(),
            'body'      => $bodyContent,
            'parent_id' => $parentId,
            'file_path' => $path,
        ]);

        // ریست بر اساس فرمی که سابمیت شده است
        if ($parentId) {
            unset($this->replyBody[$parentId]);
            $this->replyTo = null; // بستن فرم پاسخ
        } else {
            $this->reset(['body', 'file', 'replyTo']);
        }

        $this->dispatch('commentAdded'); // رفرش خودکار لیست
    }

    public function setReply($id)
    {
        $this->replyTo = $id;
        $this->replyBody[$id] = ''; // مطمئن شویم فیلد پاسخ آماده است
        $this->dispatch('scroll-to-reply-form');
    }

    public function cancelReply()
    {
        $this->replyTo = null;
    }

    public function render()
    {
        // ثبت دیده شدن برای همه کامنت‌های سطح اول (برای کارایی)
        if(Auth::check()) {
            $commentIds = \App\Models\TaskComment::where('task_id', $this->taskId)->whereNull('parent_id')->pluck('id');
            foreach($commentIds as $id) {
                \App\Models\TaskCommentView::firstOrCreate([
                    'task_comment_id' => $id,
                    'user_id' => auth()->id(),
                ]);
            }
        }

        // 💡 واکشی کامنت‌های سطح اول (parent_id = null)
        $comments = \App\Models\TaskComment::with([
            'user',
            // 🛑 واکشی پاسخ‌ها با عمق مشخص (برای نمایش سریع)
            'replies' => fn($q) => $q->with('user')->orderBy('created_at', 'asc'),
        ])
            ->where('task_id', $this->taskId)
            ->whereNull('parent_id')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('livewire.admin.task.comment-list', [
            'comments' => $comments,
            'replyTo'  => $this->replyTo,
        ]);
    }
}
