<?php

namespace App\Livewire\Admin\Task;

use App\Models\Project;
use App\Models\Task;
use App\Models\Section;
use App\Models\User;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Morilog\Jalali\Jalalian;

#[Layout('admin.master')]
class CreateTask extends Component
{
    public Project $project;

    // فیلدهای فرم
    public $section_id;
    public $assigned_to;
    public $title;
    public $description;
    public $status = 'new';
    public $priority = 'normal';
    public $type = 'task';
    public $start_date;     // شمسی، رشته انگلیسی: 1402/01/01
    public $due_date;       // شمسی، رشته انگلیسی: 1402/01/15
    public $dependency_id;
    public $parent_id;
    public $progress_percent = 0;

    // لیست‌های کمکی
    public $sections = [];
    public $users = [];
    public $tasks = [];

    protected $rules = [
        'section_id'       => 'nullable|integer|exists:sections,id',
        'assigned_to'      => 'nullable|integer|exists:users,id',
        'dependency_id'    => 'nullable|integer|exists:tasks,id',
        'parent_id'        => 'nullable|integer|exists:tasks,id',

        'title'            => 'required|string|max:255',
        'description'      => 'nullable|string|max:1000',
        'status'           => 'required|in:new,in_progress,done,on_hold',
        'priority'         => 'required|in:low,normal,high,critical',
        'type'             => 'required|in:task,issue,bug,feature',

        'start_date'       => 'nullable|string',
        'due_date'         => 'nullable|string',

        'progress_percent' => 'nullable|integer|min:0|max:100',
    ];

    protected $messages = [
        'title.required'   => 'عنوان تسک الزامی است.',
        'due_date.after_or_equal' => 'تاریخ پایان باید بعد از تاریخ شروع باشد.',
    ];

    public function mount(Project $project)
    {
        $user = Auth::user();
        abort_unless($user, 403);

        abort_unless(
            $user->isAdmin() || $user->projects()->where('project_id', $project->id)->exists(),
            403,
            'شما به این پروژه دسترسی ندارید.'
        );

        $this->project = $project;
        $this->sections = $project->sections;
        $this->users = $this->project->members()
            ->select('users.id', 'users.name') // ✅ اگر select می‌زنید، حتماً alias بزنید
            ->get();

     //  $this->users = $project->members()->select('id', 'name')->get();
        $this->tasks = $project->tasks()->select('id', 'title')->get();
    }

    /**
     * تبدیل اعداد فارسی و عربی به انگلیسی
     */
    private function convertNumbersToEnglish(string $str): string
    {
        $persian = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹'];
        $arabic  = ['٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
        $english = ['0','1','2','3','4','5','6','7','8','9'];

        return str_replace(
            array_merge($persian, $arabic),
            $english,
            trim($str)
        );
    }

    /**
     * تجزیه تاریخ شمسی (Y/m/d) به Carbon
     */
    private function parseJalaliDate(?string $jalaliDate): ?\Carbon\Carbon
    {
        if (blank($jalaliDate)) {
            return null;
        }

        try {
            $cleanDate = $this->convertNumbersToEnglish($jalaliDate);
            return Jalalian::fromFormat('Y/m/d', $cleanDate)->toCarbon();
        } catch (\Exception $e) {
            Log::warning('Invalid Jalali date received', [
                'input' => $jalaliDate,
                'cleaned' => $this->convertNumbersToEnglish($jalaliDate ?? ''),
                'error' => $e->getMessage()
            ]);
            $this->addError('start_date', 'فرمت تاریخ نامعتبر است. لطفاً از دیت‌پیکر استفاده کنید.');
            return null;
        }
    }

    public function saveTask()
    {
        $validated = $this->validate();

        $startDate = $this->parseJalaliDate($validated['start_date']);
        $dueDate   = $this->parseJalaliDate($validated['due_date']);

        // اعتبارسنجی منطقی تاریخ‌ها
        if ($startDate && $dueDate && $dueDate->lessThan($startDate)) {
            $this->addError('due_date', 'تاریخ پایان باید بعد از تاریخ شروع باشد.');
            return;
        }

        // تبدیل فیلدهای خالی به null برای ستون‌های integer
        $section_id    = empty($validated['section_id'])    ? null : (int) $validated['section_id'];
        $assigned_to   = empty($validated['assigned_to'])   ? null : (int) $validated['assigned_to'];
        $dependency_id = empty($validated['dependency_id']) ? null : (int) $validated['dependency_id'];
        $parent_id     = empty($validated['parent_id'])     ? null : (int) $validated['parent_id'];

        try {
            $task = Task::create([
                'project_id'       => $this->project->id,
                'created_by'       => Auth::id(),
                'title'            => $validated['title'],
                'description'      => $validated['description'],
                'status'           => $validated['status'],
                'priority'         => $validated['priority'],
                'type'             => $validated['type'],
                'progress_percent' => $validated['progress_percent'] ?? 0,
                'section_id'       => $section_id,
                'assigned_to'      => $assigned_to,
                'dependency_id'    => $dependency_id,
                'parent_id'        => $parent_id,
                'start_date'       => $startDate,
                'due_date'         => $dueDate,
            ]);

            session()->flash('message', '✅ تسک جدید با موفقیت ثبت شد.');

            // ریست فرم
            $this->reset([
                'section_id', 'assigned_to', 'title', 'description', 'status',
                'priority', 'type', 'start_date', 'due_date', 'dependency_id',
                'parent_id', 'progress_percent',
            ]);
            $this->status = 'new';
            $this->priority = 'normal';
            $this->type = 'task';
            $this->progress_percent = 0;

            // رفرش لیست
            $this->tasks = $this->project->tasks()
                ->select('id', 'title', 'due_date', 'assigned_to')
                ->get();

            $this->dispatch('taskCreated');

        } catch (\Exception $e) {
            Log::error('Task creation failed', [
                'project_id' => $this->project->id,
                'user_id' => Auth::id(),
                'input' => $validated,
                'exception' => $e
            ]);
            session()->flash('error', '❌ خطایی در ثبت تسک رخ داد. لطفاً دوباره تلاش کنید.');
        }
    }

    public function render()
    {
        return view('livewire.admin.task.create-task');
    }
}
