<?php
// FILE: App/Livewire/Admin/Task/ListTasks.php (کلاس PHP)
namespace App\Livewire\Admin\Task;

use App\Models\Task;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;

class ListTasks extends Component
{
    use WithPagination;
    protected $paginationTheme = 'bootstrap';

    public $projectId;
    public $search = '';
    public $status = '';
    public $priority = '';
    public $viewType = 'table'; // 'table' یا 'card'
    public $perPage = 12;

    // Map status/priority values to custom labels for the view
    protected $labels = [
        'statuses' => [
            'new' => 'جدید',
            'in_progress' => 'درحال انجام',
            'done' => 'انجام شده',
            'on_hold' => 'متوقف',
        ],
        'priorities' => [
            'low' => 'کم',
            'normal' => 'متوسط',
            'high' => 'زیاد',
            'critical' => 'بحرانی',
        ],
    ];

    public function mount($project)
    {
        $this->projectId = $project;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedStatus()
    {
        $this->resetPage();
    }

    public function updatedPriority()
    {
        $this->resetPage();
    }

    // 💡 متد هوشمند برای حذف تمام فیلترها
    public function resetFilters()
    {
        $this->search = '';
        $this->status = '';
        $this->priority = '';
        $this->resetPage();
    }

    public function toggleView()
    {
        $this->viewType = $this->viewType === 'table' ? 'card' : 'table';
    }

    public function render()
    {
        // Eager load assignedUser for better performance
        $query = Task::where('project_id', $this->projectId)
            ->with('assignedUser');

        if ($this->search) {
            $query->where(function($q) {
                $q->where('title', 'like', "%{$this->search}%")
                    ->orWhere('description', 'like', "%{$this->search}%");
            });
        }
        if ($this->status) {
            $query->where('status', $this->status);
        }
        if ($this->priority) {
            $query->where('priority', $this->priority);
        }

        $totalCount = Task::where('project_id', $this->projectId)->count();
        $filteredCount = (clone $query)->count();

        // 💡 Optimized Ordering: High priority/In Progress tasks first
        $tasks = $query->orderByRaw("FIELD(status, 'in_progress', 'new', 'on_hold', 'done')")
            ->orderByRaw("FIELD(priority, 'critical', 'high', 'normal', 'low')")
            ->orderBy('due_date')
            ->paginate($this->perPage)
            // Map labels to tasks after pagination
            ->through(function ($task) {
                // Ensure correct label is attached for use in Blade
                $task->status_label = $this->labels['statuses'][$task->status] ?? Str::title($task->status);
                return $task;
            });

        return view('livewire.admin.task.list-tasks', [
            'tasks' => $tasks,
            'viewType' => $this->viewType,
            'projectId' => $this->projectId,
            'status'    => $this->status,
            'priority'  => $this->priority,
            'search'    => $this->search,
            'totalCount' => $totalCount,
            'filteredCount' => $filteredCount,
            'labels' => $this->labels,
        ])->layout('admin.master');
    }
}
