<?php

namespace App\Livewire\Admin\Task;

use Livewire\Component;
use App\Models\Task;
use App\Models\Project;
use App\Models\Section;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Str;

class TaskEdit extends Component
{

    public $taskId;
    public $projectId;

    public $section_id;
    public $assigned_to;
    public $title;
    public $description;
    public $status;
    public $priority;
    public $type;
    public $start_date; // این فیلدها در Livewire به صورت رشته‌های شمسی نگهداری می‌شوند.
    public $due_date;   // هنگام ذخیره در دیتابیس باید به میلادی تبدیل شوند (که فرض شده در متد save انجام می‌شود).
    public $dependency_id;
    public $parent_id; // فیلد جدید برای تسک والد
    public $progress_percent;
    public $project;

    public $sections = [];
    public $users = [];
    public $tasks = [];

    protected $rules = [
        'section_id' => 'nullable|exists:sections,id',
        'assigned_to' => 'nullable|exists:users,id',
        'title' => 'required|string|max:255',
        'description' => 'nullable|string',
        'status' => 'required|in:new,in_progress,done,on_hold',
        // 💡 اضافه شدن اولویت بحرانی
        'priority' => 'required|in:low,normal,high,critical',
        'type' => 'required|in:task,issue,bug,feature',
        'start_date' => 'nullable|string', // رشته شمسی
        'due_date' => 'nullable|string', // رشته شمسی
        'dependency_id' => 'nullable|exists:tasks,id',
        'parent_id' => 'nullable|exists:tasks,id',
        'progress_percent' => 'nullable|integer|min:0|max:100',
    ];

    protected $messages = [
        'due_date.after_or_equal' => 'تاریخ پایان باید بعد از تاریخ شروع باشد.',
    ];

    public function mount($project, $task)
    {
        $this->projectId = $project;
        $this->taskId = $task;

        $taskModel = Task::findOrFail($task);
        $this->project = Project::findOrFail($project);

        // 🛑 FIX: تبدیل تاریخ‌های میلادی دیتابیس به فرمت YYYY-MM-DD برای HTML Input
        // اگر DatePicker شما مستقیما میلادی YYYY-MM-DD را می‌پذیرد:
        $this->start_date        = $taskModel->start_date ? Carbon::parse($taskModel->start_date)->toDateString() : null;
        $this->due_date          = $taskModel->due_date ? Carbon::parse($taskModel->due_date)->toDateString() : null;

        // اگر DatePicker شما شمسی کار می‌کند، اینجا باید تبدیل به شمسی انجام شود:
        // $this->start_date = $taskModel->start_date ? verta($taskModel->start_date)->format('Y/m/d') : null;
        // $this->due_date = $taskModel->due_date ? verta($taskModel->due_date)->format('Y/m/d') : null;


        $this->section_id        = $taskModel->section_id;
        $this->assigned_to       = $taskModel->assigned_to;
        $this->title             = $taskModel->title;
        $this->description       = $taskModel->description;
        $this->status            = $taskModel->status;
        $this->priority          = $taskModel->priority;
        $this->type              = $taskModel->type;
        $this->dependency_id     = $taskModel->dependency_id;
        $this->parent_id         = $taskModel->parent_id; // لود کردن والد
        $this->progress_percent  = $taskModel->progress_percent;


        $this->sections = Section::where('project_id', $this->projectId)->get();
        $this->users = $this->project->members()->select('users.id', 'users.name')->get();
        // 💡 حذف تسک فعلی از لیست وابستگی و والد
        $this->tasks = Task::where('project_id', $this->projectId)->where('id', '<>', $task)->get();
    }

    public function updateTask()
    {
        // 🛑 نکته: اگر ورودی‌های start_date و due_date شمسی هستند، باید قبل از اعتبارسنجی، آن‌ها را به میلادی تبدیل کنید یا یک قانون اعتبارسنجی سفارشی برای تاریخ شمسی بنویسید.
        // در این مثال، فرض می‌کنیم یا ورودی‌ها میلادی هستند، یا شما از اعتبارسنجی پس از تبدیل استفاده می‌کنید.
        $this->validate();

        $task = Task::findOrFail($this->taskId);
        $task->update([
            'section_id'       => $this->section_id,
            'assigned_to'      => $this->assigned_to,
            'title'            => $this->title,
            'description'      => $this->description,
            'status'           => $this->status,
            'priority'         => $this->priority,
            'type'             => $this->type,
            'start_date'       => $this->start_date,
            'due_date'         => $this->due_date,
            'dependency_id'    => $this->dependency_id,
            'parent_id'        => $this->parent_id, // ذخیره والد
            'progress_percent' => $this->progress_percent,
        ]);

        session()->flash('message', 'تسک با موفقیت ویرایش شد!');
    }


    public function render()
    {
        return view('livewire.admin.task.task-edit')->layout('admin.master');
    }
}
