<?php

namespace App\Livewire\Admin\Transmittal;

use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

use App\Models\Transmittal;
use App\Models\TransmittalDocument;
use App\Models\TransmittalAttachment;
use App\Models\TransmittalNumber;
use App\Models\TransmittalTemplate;
use App\Models\TransmittalRecipient;
use App\Models\TransmittalLog;
use App\Models\User;
use App\Models\Project;
use App\Models\MasterDocument;
use App\Models\DocumentRevision;
use Morilog\Jalali\Jalalian;

class CreateTransmittal extends Component
{
    use WithFileUploads, WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $step = 'form';

    public $project_id, $project_name, $isProjectSelected = false;
    public $projects = [];

    // تاریخ‌ها
    public $transmittal_date;           // میلادی برای ذخیره (Y-m-d)
    public $transmittal_date_jalali;    // شمسی برای نمایش/ورود (Y/m/d)

    public $sender_id, $receiver_id, $purpose, $description;
    public $senders = [], $recipients = [], $templates = [], $template_id = null;

    public $searchTerm = '';
    public $selected_documents = [];
    public $selected_revisions = [];

    public $attachments = [];
    public $cc_recipients = [];

    // نمایش
    public $transmittal_number_preview = 'TR-UNKNOWN-A-000';
    public $sender_name = '-', $receiver_name = '-', $cc_names = [];

    // صفحه‌بندی + سورت
    public int $docsPerPage = 10;
    public string $docSortField = 'document_number';
    public string $docSortDir   = 'asc';

    // انتخاب همۀ نتایج
    public bool $allMatchingSelected = false;

    // چیپ‌های «هدف ارسال» متداول
    public array $purposeOptions = ['برای تایید','برای اطلاع','برای اقدام','برای بایگانی'];

    protected $queryString = [
        'project_id' => ['except' => ''],
        'searchTerm' => ['except' => ''],
        'docsPage'   => ['except' => 1],
    ];

    public function mount($project_id = null)
    {
        $this->project_id = $project_id ?? request()->query('project_id');

        $this->transmittal_date = now()->format('Y-m-d');
        $this->transmittal_date_jalali = Jalalian::fromCarbon(now())->format('Y/m/d');

        $this->senders    = User::select('id','name')->orderBy('name')->get();
        $this->recipients = User::select('id','name')->orderBy('name')->get();
        $this->templates  = TransmittalTemplate::select('id','title','purpose','description','sender_id')
            ->orderBy('title')->get();

        if ($this->project_id && ($p = Project::select('id','name','code')->find($this->project_id))) {
            $this->project_name = $p->name;
            $this->isProjectSelected = true;
            $this->projects = [];
        } else {
            $this->isProjectSelected = false;
            $this->projects = Project::select('id','name','code')->orderBy('name')->get();
            $this->project_id = null;
            $this->project_name = null;
        }

        $this->refreshPreviewNumber();
        $this->refreshNames();
    }

    public function updatedTransmittalDateJalali($value)
    {
        try {
            if (!$value) { $this->transmittal_date = null; return; }
            $this->transmittal_date = Jalalian::fromFormat('Y/m/d', trim($value))->toCarbon()->format('Y-m-d');
            $this->resetErrorBag('transmittal_date_jalali');
        } catch (\Throwable $e) {
            $this->addError('transmittal_date_jalali','تاریخ نامعتبر است (الگو: ۱۴۰۳/۰۷/۲۰).');
        }
    }

    public function clearProject()
    {
        $this->reset(['project_id','project_name','isProjectSelected','selected_documents','selected_revisions','searchTerm','allMatchingSelected']);
        $this->projects = Project::select('id','name','code')->orderBy('name')->get();
        $this->resetPage('docsPage');
        $this->refreshPreviewNumber();
    }

    public function updatedProjectId()
    {
        if ($this->project_id && ($p = Project::select('id','name','code')->find($this->project_id))) {
            $this->project_name = $p->name;
            $this->isProjectSelected = true;
            $this->projects = [];
        } else {
            $this->clearProject();
            return;
        }
        $this->allMatchingSelected = false;
        $this->resetPage('docsPage');
        $this->refreshPreviewNumber();
    }

    public function updatedSearchTerm()
    {
        $this->allMatchingSelected = false;
        $this->resetPage('docsPage');
    }

    public function updatedTemplateId()
    {
        if (!$this->template_id) return;
        $tpl = $this->templates->firstWhere('id', (int)$this->template_id);
        if ($tpl) {
            $this->purpose     = $tpl->purpose;
            $this->description = $tpl->description;
            $this->sender_id   = $tpl->sender_id;
            $this->refreshNames();
        }
    }

    public function updatedSenderId(){ $this->refreshNames(); }
    public function updatedReceiverId(){ $this->refreshNames(); }
    public function updatedCcRecipients(){ $this->refreshNames(); }

    public function updatedSelectedDocuments()
    {
        $this->selected_revisions = array_intersect_key(
            $this->selected_revisions,
            array_flip($this->selected_documents)
        );
    }

    public function sortDocsBy(string $field)
    {
        $allowed = ['document_number','title'];
        if (!in_array($field, $allowed, true)) return;

        if ($this->docSortField === $field) {
            $this->docSortDir = $this->docSortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->docSortField = $field;
            $this->docSortDir   = 'asc';
        }
        $this->resetPage('docsPage');
    }

    public function toggleSelectPage(int $checked)
    {
        $pageIds = $this->documents->pluck('id')->all();
        if ($checked) {
            $this->selected_documents = array_values(array_unique(array_merge($this->selected_documents, $pageIds)));
        } else {
            $this->selected_documents = array_values(array_diff($this->selected_documents, $pageIds));
        }
        $this->updatedSelectedDocuments();
    }

    public function selectAllMatching()
    {
        if (!$this->isProjectSelected) return;
        $ids = MasterDocument::where('project_id',$this->project_id)
            ->when($this->searchTerm, function($q){
                $s = trim($this->searchTerm);
                $q->where(function($qq) use ($s){
                    $qq->where('document_number','like',"%{$s}%")
                        ->orWhere('title','like', "%{$s}%");
                });
            })
            ->pluck('id')
            ->all();

        $this->selected_documents = $ids;
        $this->allMatchingSelected = true;
        $this->updatedSelectedDocuments();
    }

    public function clearSelection()
    {
        $this->selected_documents = [];
        $this->selected_revisions = [];
        $this->allMatchingSelected = false;
    }

    protected function refreshNames()
    {
        $this->sender_name   = $this->sender_id   ? (User::find($this->sender_id)->name ?? '-') : '-';
        $this->receiver_name = $this->receiver_id ? (User::find($this->receiver_id)->name ?? '-') : '-';
        $this->cc_names      = $this->cc_recipients ? User::whereIn('id',$this->cc_recipients)->pluck('name')->toArray() : [];
    }

    protected function refreshPreviewNumber()
    {
        $this->transmittal_number_preview = $this->peekNextNumber();
    }

    protected function peekNextNumber(): string
    {
        if (!$this->project_id) return 'TR-UNKNOWN-A-000';

        $project = Project::select('id','name','code')->find($this->project_id);
        if (!$project) return 'TR-UNKNOWN-A-000';

        $code = $this->normalizeProjectCode($project->code ?: $this->initials($project->name));
        $last = optional(TransmittalNumber::where('project_code',$code)->first())->last_number ?? 0;
        return "TR-{$code}-A-".str_pad($last+1, 3, '0', STR_PAD_LEFT);
    }

    protected function allocateNextNumber(string $code): string
    {
        return DB::transaction(function() use ($code){
            $rec = TransmittalNumber::lockForUpdate()->firstOrCreate(['project_code'=>$code], ['last_number'=>0]);
            $rec->increment('last_number');
            $n = $rec->last_number;
            return "TR-{$code}-A-".str_pad($n,3,'0',STR_PAD_LEFT);
        });
    }

    protected function initials(string $name): string
    {
        $parts = preg_split('/\s+/', trim($name)) ?: [];
        $ini = '';
        foreach ($parts as $w) $ini .= Str::substr($w,0,1);
        return strtoupper(Str::substr($ini,0,4));
    }

    protected function normalizeProjectCode(string $code): string
    {
        $code = preg_replace('/[^A-Za-z0-9]/','', $code);
        $code = strtoupper(Str::substr($code,0,4));
        return $code ?: 'UNKN';
    }

    /** ✅ اصلاح شده: حذف 'revision' از select و لود کردن revisions */
    public function getDocumentsProperty()
    {
        if (!$this->isProjectSelected) {
            return MasterDocument::query()->whereRaw('1=0')
                ->paginate($this->docsPerPage, ['*'], 'docsPage');
        }

        return MasterDocument::with(['revisions:id,master_document_id,revision,created_at'])
            ->select('id','project_id','document_number','title','description') // ✅ 'revision' حذف شد
            ->where('project_id',$this->project_id)
            ->when($this->searchTerm, function($q){
                $s = trim($this->searchTerm);
                $q->where(function($qq) use ($s){
                    $qq->where('document_number','like',"%{$s}%")
                        ->orWhere('title','like', "%{$s}%");
                });
            })
            ->orderBy($this->docSortField, $this->docSortDir)
            ->paginate($this->docsPerPage, ['*'], 'docsPage');
    }

    /** ✅ اصلاح شده: استفاده از latest revision به جای $doc->revision */
    public function getPreviewDocsProperty(): array
    {
        if (!$this->selected_documents || !count($this->selected_documents)) return [];

        // ✅ لود کردن revisions برای مدارک انتخاب‌شده
        $docs = MasterDocument::select('id','document_number','title')
            ->with('revisions:id,master_document_id,revision') // ✅ لود رویژن‌ها
            ->whereIn('id',$this->selected_documents)->get()->keyBy('id');

        $revIds = array_values(array_filter($this->selected_revisions));
        $revs   = $revIds ? DocumentRevision::select('id','master_document_id','revision','created_at')
            ->whereIn('id',$revIds)->get()->keyBy('id') : collect();

        $out = [];
        foreach ($this->selected_documents as $docId){
            $d = $docs[$docId] ?? null;
            if(!$d) continue;

            $rid = $this->selected_revisions[$docId] ?? null;
            $rev = $rid ? ($revs[$rid] ?? null) : null;

            // ✅ استفاده از آخرین رویژن اگر رویژنی انتخاب نشده
            $latestRev = $d->revisions->first();
            $fallbackRevision = $latestRev ? $latestRev->revision : 'بدون رویژن';

            $out[] = [
                'doc_id'        => $docId,
                'revision_id'   => $rid,
                'number'        => $d->document_number,
                'title'         => $d->title,
                'revision_label'=> $rev ? $rev->revision : $fallbackRevision,
            ];
        }
        return $out;
    }

    public function nextStep()
    {
        if ($this->transmittal_date_jalali && !$this->transmittal_date) {
            $this->updatedTransmittalDateJalali($this->transmittal_date_jalali);
        }

        $this->validate([
            'project_id'         => 'required|exists:projects,id',
            'transmittal_date'   => 'required|date',
            'sender_id'          => 'required|exists:users,id',
            'receiver_id'        => 'required|exists:users,id',
            'purpose'            => 'required|string|max:100',
            'selected_documents' => 'required|array|min:1',
        ]);

        $this->step = 'preview';
        $this->refreshNames();
    }

    public function previousStep(){ $this->step = 'form'; }

    public function removeAttachment($index)
    {
        unset($this->attachments[$index]);
        $this->attachments = array_values($this->attachments);
    }

    /** ✅ اصلاح شده: ذخیره با استفاده از رویژن‌ها */
    public function saveTransmittal()
    {
        if ($this->transmittal_date_jalali && !$this->transmittal_date) {
            $this->updatedTransmittalDateJalali($this->transmittal_date_jalali);
        }

        $this->validate([
            'project_id'         => 'required|exists:projects,id',
            'transmittal_date'   => 'required|date',
            'sender_id'          => 'required|exists:users,id',
            'receiver_id'        => 'required|exists:users,id',
            'purpose'            => 'required|string|max:100',
            'selected_documents' => 'required|array|min:1',
            'cc_recipients'      => 'nullable|array',
            'cc_recipients.*'    => 'exists:users,id',
            'attachments'        => 'nullable|array|max:5',
            'attachments.*'      => 'file|mimes:pdf,jpg,jpeg,png,doc,docx,xls,xlsx|max:50480',
        ]);

        $project = Project::select('id','name','code')->find($this->project_id);
        if (!$project) {
            session()->flash('error','پروژه نامعتبر است.');
            return;
        }
        $code = $this->normalizeProjectCode($project->code ?: $this->initials($project->name));

        DB::beginTransaction();
        try {
            $realNumber = $this->allocateNextNumber($code);

            $trans = Transmittal::create([
                'project_id'         => $this->project_id,
                'transmittal_number' => $realNumber,
                'sent_at'            => $this->transmittal_date,
                'sender_id'          => $this->sender_id,
                'receiver_id'        => $this->receiver_id,
                'purpose'            => $this->purpose,
                'description'        => $this->description,
                'status'             => 'ارسال شده',
            ]);

            TransmittalRecipient::create([
                'transmittal_id' => $trans->id,
                'user_id'        => $this->receiver_id,
                'role'           => 'to',
            ]);

            foreach ($this->cc_recipients ?? [] as $cc) {
                TransmittalRecipient::create([
                    'transmittal_id' => $trans->id,
                    'user_id'        => $cc,
                    'role'           => 'cc',
                ]);
            }

            foreach ($this->attachments ?? [] as $attachment) {
                $path = $attachment->store('transmittals', 'public');
                TransmittalAttachment::create([
                    'transmittal_id' => $trans->id,
                    'file_name'      => $attachment->getClientOriginalName(),
                    'file_path'      => $path,
                    'size'           => $attachment->getSize(),
                    'mime'           => $attachment->getMimeType(),
                ]);
            }

            // ✅ اصلاح: لود کردن revisions برای مدارک
            $docs = MasterDocument::select('id','description')
                ->with('revisions:id,master_document_id,revision,remark') // ✅ لود رویژن‌ها
                ->whereIn('id',$this->selected_documents)->get()->keyBy('id');

            $revIds = array_values(array_filter($this->selected_revisions));
            $revs   = $revIds ? DocumentRevision::select('id','master_document_id','revision','remark')
                ->whereIn('id',$revIds)->get()->keyBy('id') : collect();

            foreach ($this->selected_documents as $docId) {
                $doc = $docs[$docId] ?? null;
                if (!$doc) throw new \RuntimeException("مدرک نامعتبر: {$docId}");

                $revId = $this->selected_revisions[$docId] ?? null;
                $rev   = $revId ? ($revs[$revId] ?? null) : null;
                if ($revId && !$rev) throw new \RuntimeException("رویژن نامعتبر برای مدرک {$docId}");

                // ✅ دریافت آخرین رویژن برای مقادیر پیش‌فرض
                $latestRev = $doc->revisions->first();
                $docRevision = $rev ? $rev->revision : ($latestRev ? $latestRev->revision : null);
                $docDescription = $rev ? $rev->remark : ($latestRev ? $latestRev->remark : $doc->description);

                TransmittalDocument::create([
                    'transmittal_id'      => $trans->id,
                    'master_document_id'  => $docId,
                    'revision_id'         => $revId,
                    'revision'            => $docRevision,
                    'description'         => $docDescription,
                ]);
            }

            TransmittalLog::create([
                'transmittal_id' => $trans->id,
                'user_id'        => auth()->id(),
                'action'         => 'created',
                'description'    => 'ترنسمیتال ایجاد شد.',
            ]);

            DB::commit();

            session()->flash('success','ترنسمیتال با موفقیت ثبت شد.');
            return redirect()->route('transmittals.show', $trans);

        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Transmittal save error: '.$e->getMessage(), ['trace'=>$e->getTraceAsString()]);
            session()->flash('error','خطا در ذخیره ترنسمیتال: '.$e->getMessage());
        }
    }

    public function updated($prop)
    {
        if (in_array($prop, ['project_id','searchTerm','docSortField','docSortDir'])) {
            $this->resetPage('docsPage');
        }
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.transmittal.create-transmittal', [
            'documents'   => $this->documents,
            'previewDocs' => $this->previewDocs,
        ]);
    }
}
