<?php

namespace App\Livewire\Admin\Transmittal;

use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Transmittal;
use App\Models\Project;
use Illuminate\Support\Facades\Response;
use Barryvdh\DomPDF\Facade\Pdf; // مطمئن شوید dompdf نصب است

class TransmittalList extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    // فیلترها/سورت
    public string $search = '';
    public ?int $project_id = null;
    public ?string $status = null;
    public int $perPage = 10;

    public string $sortField = 'created_at';
    public string $sortDir   = 'desc';

    // تب‌ها و شمارنده‌ها
    public array $statuses = ['ارسال شده','دریافت شده','در انتظار','رد شده'];
    public array $statusCounts = ['همه' => 0, 'ارسال شده'=>0,'دریافت شده'=>0,'در انتظار'=>0,'رد شده'=>0];

    // مودال
    public ?\App\Models\Transmittal $modalTr = null;

    protected $queryString = [
        'search'     => ['except' => ''],
        'project_id' => ['except' => ''],
        'status'     => ['except' => ''],
        'perPage'    => ['except' => 10],
        'sortField'  => ['except' => 'created_at'],
        'sortDir'    => ['except' => 'desc'],
        'page'       => ['except' => 1],
    ];

    // ریست صفحه روی تغییر فیلترها

    private function utf8Sanitize(null|string|int|float $v): string
    {
        if (is_null($v)) return '';
        $s = (string)$v;

        // اگر متن UTF-8 نیست، تبدیلش کن
        if (!mb_check_encoding($s, 'UTF-8')) {
            $enc = mb_detect_encoding($s, ['UTF-8','Windows-1256','ISO-8859-1','ASCII'], true) ?: 'UTF-8';
            $s = mb_convert_encoding($s, 'UTF-8', $enc);
        }

        // حذف کاراکترهای کنترلی (به‌جز \n\t\r اگر لازم دارید نگه دارید)
        $s = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $s);

        return $s;
    }
    public function updatingSearch(){ $this->resetPage(); }
    public function updatingProjectId(){ $this->resetPage(); }
    public function updatingStatus(){ $this->resetPage(); }
    public function updatingPerPage(){ $this->resetPage(); }

    public function setStatus(?string $st): void
    {
        $this->status = $st ?: null;
        $this->resetPage();
    }

    public function sortBy(string $field): void
    {
        $allowed = ['transmittal_number','project_name','sent_at','created_at','status'];
        if (!in_array($field, $allowed, true)) return;

        if ($this->sortField === $field) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDir   = 'asc';
        }
        $this->resetPage();
    }

    public function clearFilters(): void
    {
        $this->reset(['search','project_id','status']);
        $this->resetPage();
    }

    public function showModal(int $id): void
    {
        $this->modalTr = Transmittal::with([
            'project:id,name,code',
            'sender:id,name',
            'receiver:id,name',
            'attachments:id,transmittal_id,file_name,file_path,size',
            'documents.masterDocument:id,document_number,title',
        ])->findOrFail($id);

        $this->dispatch('open-transmittal-modal');
    }

    public function closeModal(): void
    {
        $this->dispatch('close-transmittal-modal');
        $this->modalTr = null;
    }

    /** کوئری پایه فیلترشده (برای لیست و اکسپورت‌ها) */
    protected function baseFilteredQuery()
    {
        return Transmittal::query()
            ->when($this->project_id, fn($q) => $q->where('project_id', $this->project_id))
            ->when($this->status,     fn($q) => $q->where('status', $this->status))
            ->when($this->search, function($q){
                $s = trim($this->search);
                $q->where(function($qq) use ($s){
                    $qq->where('transmittal_number','like',"%{$s}%")
                        ->orWhere('purpose','like',"%{$s}%")
                        ->orWhereHas('project', fn($p)=>$p->where('name','like',"%{$s}%"))
                        ->orWhereHas('sender',  fn($p)=>$p->where('name','like',"%{$s}%"))
                        ->orWhereHas('receiver',fn($p)=>$p->where('name','like',"%{$s}%"));
                });
            });
    }

    /** کوئری لیست با روابط و سورت */
    protected function listQuery()
    {
        $q = $this->baseFilteredQuery()
            ->with(['project:id,name','sender:id,name','receiver:id,name','attachments:id,transmittal_id,file_path'])
            ->withCount('documents');

        if ($this->sortField === 'project_name') {
            $q->leftJoin('projects','projects.id','=','transmittals.project_id')
                ->select('transmittals.*')
                ->orderBy('projects.name', $this->sortDir);
        } else {
            $q->orderBy($this->sortField, $this->sortDir);
        }
        return $q;
    }

    /** اکسپورت CSV با فیلترهای فعلی */
    public function exportCsv()
    {
        $filename = 'transmittals_'.now()->format('Ymd_His').'.csv';

        $query = $this->baseFilteredQuery()
            ->with(['project:id,name','sender:id,name','receiver:id,name'])
            ->withCount('documents')
            ->orderBy($this->sortField === 'project_name' ? 'created_at' : $this->sortField, $this->sortDir);

        return Response::streamDownload(function() use ($query) {
            // BOM برای نمایش درست فارسی در Excel
            echo "\xEF\xBB\xBF";
            $out = fopen('php://output', 'w');

            fputcsv($out, [
                $this->utf8Sanitize('شماره'),
                $this->utf8Sanitize('پروژه'),
                $this->utf8Sanitize('فرستنده'),
                $this->utf8Sanitize('گیرنده'),
                $this->utf8Sanitize('تاریخ'),
                $this->utf8Sanitize('هدف'),
                $this->utf8Sanitize('تعداد مدارک'),
                $this->utf8Sanitize('وضعیت'),
            ]);

            $query->chunkById(500, function($rows) use ($out) {
                foreach ($rows as $tr) {
                    $date = jdate($tr->sent_at ?? $tr->created_at)->format('Y/m/d');

                    fputcsv($out, [
                        $this->utf8Sanitize($tr->transmittal_number),
                        $this->utf8Sanitize($tr->project->name ?? '-'),
                        $this->utf8Sanitize($tr->sender->name ?? '-'),
                        $this->utf8Sanitize($tr->receiver->name ?? '-'),
                        $this->utf8Sanitize($date),
                        $this->utf8Sanitize($tr->purpose ?? '-'),
                        $this->utf8Sanitize($tr->documents_count ?? $tr->documents()->count()),
                        $this->utf8Sanitize($tr->status),
                    ]);
                }
            });

            fclose($out);
        }, $filename, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Cache-Control' => 'no-store, no-cache, must-revalidate, max-age=0',
            'Pragma' => 'no-cache',
        ]);
    }

//    public function exportCsv()
//    {
//        $filename = 'transmittals_'.now()->format('Ymd_His').'.csv';
//
//        $query = $this->baseFilteredQuery()
//            ->with(['project:id,name','sender:id,name','receiver:id,name'])
//            ->withCount('documents')
//            ->orderBy($this->sortField === 'project_name' ? 'created_at' : $this->sortField, $this->sortDir);
//
//        return Response::streamDownload(function() use ($query){
//            // BOM برای فارسی در اکسل
//            echo "\xEF\xBB\xBF";
//            $out = fopen('php://output', 'w');
//            fputcsv($out, ['شماره','پروژه','فرستنده','گیرنده','تاریخ','هدف','تعداد مدارک','وضعیت']);
//
//            $query->chunkById(500, function($rows) use ($out){
//                foreach ($rows as $tr) {
//                    $date = jdate($tr->sent_at ?? $tr->created_at)->format('Y/m/d');
//                    fputcsv($out, [
//                        $tr->transmittal_number,
//                        $tr->project->name ?? '-',
//                        $tr->sender->name ?? '-',
//                        $tr->receiver->name ?? '-',
//                        $date,
//                        $tr->purpose ?? '-',
//                        $tr->documents_count ?? $tr->documents()->count(),
//                        $tr->status,
//                    ]);
//                }
//            });
//
//            fclose($out);
//        }, $filename, [
//            'Content-Type' => 'text/csv; charset=UTF-8',
//        ]);
//    }

    /** اکسپورت PDF (نیازمند barryvdh/laravel-dompdf) */
    public function exportPdf()
    {
        $items = $this->baseFilteredQuery()
            ->with(['project:id,name','sender:id,name','receiver:id,name'])
            ->withCount('documents')
            ->orderBy($this->sortField === 'project_name' ? 'created_at' : $this->sortField, $this->sortDir)
            ->limit(5000)
            ->get();

        // سطرهای «تمیز» برای PDF (فقط آرایه ساده؛ نه مدل)
        $rows = $items->map(function($tr){
            $date = jdate($tr->sent_at ?? $tr->created_at)->format('Y/m/d');

            return [
                'number'    => $this->utf8Sanitize($tr->transmittal_number),
                'project'   => $this->utf8Sanitize($tr->project->name ?? '-'),
                'sender'    => $this->utf8Sanitize($tr->sender->name ?? '-'),
                'receiver'  => $this->utf8Sanitize($tr->receiver->name ?? '-'),
                'date'      => $this->utf8Sanitize($date),
                'purpose'   => $this->utf8Sanitize($tr->purpose ?? '-'),
                'docs'      => (int)($tr->documents_count ?? $tr->documents()->count()),
                'status'    => $this->utf8Sanitize($tr->status),
            ];
        })->all();

        $pdf = Pdf::loadView('exports.transmittals-pdf', [
            'rows'       => $rows,
            'printed_at' => jdate(now())->format('Y/m/d H:i'),
            'title'      => 'گزارش ترنسمیتال‌ها',
        ])->setPaper('a4', 'portrait');

        return $pdf->download('transmittals_'.now()->format('Ymd_His').'.pdf');
    }

//    public function exportPdf()
//    {
//        $items = $this->baseFilteredQuery()
//            ->with(['project:id,name','sender:id,name','receiver:id,name'])
//            ->withCount('documents')
//            ->orderBy($this->sortField === 'project_name' ? 'created_at' : $this->sortField, $this->sortDir)
//            ->limit(5000) // سقف منطقی برای PDF
//            ->get();
//
//        $pdf = Pdf::loadView('exports.transmittals-pdf', [
//            'items' => $items,
//            'printed_at' => jdate(now())->format('Y/m/d H:i'),
//            'title' => 'گزارش ترنسمیتال‌ها',
//        ])->setPaper('a4','portrait');
//
//        return $pdf->download('transmittals_'.now()->format('Ymd_His').'.pdf');
//    }

    #[Layout('admin.master')]
    public function render()
    {
        $projects = Project::select('id','name')->orderBy('name')->get();

        // شمارنده تب‌ها با فیلترهای جاری (به‌جز وضعیت)
        $base = Transmittal::query()
            ->when($this->project_id, fn($q) => $q->where('project_id', $this->project_id))
            ->when($this->search, function($q){
                $s = trim($this->search);
                $q->where(function($qq) use ($s){
                    $qq->where('transmittal_number','like',"%{$s}%")
                        ->orWhere('purpose','like',"%{$s}%")
                        ->orWhereHas('project', fn($p)=>$p->where('name','like',"%{$s}%"))
                        ->orWhereHas('sender',  fn($p)=>$p->where('name','like',"%{$s}%"))
                        ->orWhereHas('receiver',fn($p)=>$p->where('name','like',"%{$s}%"));
                });
            });

        $this->statusCounts = ['همه' => (clone $base)->count()];
        foreach ($this->statuses as $st) {
            $this->statusCounts[$st] = (clone $base)->where('status',$st)->count();
        }

        $transmittals = $this->listQuery()->paginate($this->perPage)->withQueryString();

        return view('livewire.admin.transmittal.transmittal-list', [
            'transmittals'  => $transmittals,
            'projects'      => $projects,
            'statuses'      => $this->statuses,
            'statusCounts'  => $this->statusCounts,
        ]);
    }
}
