<?php

namespace App\Livewire\Admin\Users;

use App\Models\User;
use App\Models\Department;
use App\Models\Company;
use App\Models\Role; // 💡 مدل Role اضافه شد
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Gate; // 💡 برای گارد امنیتی

#[Layout('admin.master')]
class AddUser extends Component
{
    use WithPagination, WithFileUploads;

    // فرم
    public $name, $ncode, $email, $mobile,
        $password, $password_confirmation,
        $role = 'user', $gender = 'male', $status = 'active',
        $degree, $company_id, $department_id,
        $image, $sign, $internal_number;

    // حالت/ویرایش
    public $editMode = false, $userId, $currentImage, $currentSign;

    // لیست‌ها
    public array $degrees  = ['دیپلم','کاردانی','کارشناسی','کارشناسی ارشد','دکترا','بدون مدرک'];
    public array $roles    = ['admin','manager','editor','user','guest'];
    public array $statuses = ['active','inactive'];

    // فیلتر/جستجو
    public string $search = '';
    public string $filterRole = '';
    public string $filterStatus = '';
    public ?int $filterCompanyId = null;
    public ?int $filterDepartmentId = null;

    // وابسته‌ها
    public $departments = [];

    // Modal تغییر رمز
    public $resetPasswordUserId;
    public $newPassword;
    public $newPasswordConfirmation;

    protected $paginationTheme = 'bootstrap';

    // سورت
    public string $sortField = 'created_at';
    public string $sortDirection = 'desc';
    protected array $sortable = ['name','email','mobile','internal_number','role','status','last_login_at','created_at'];

    // صفحه‌بندی
    public int $perPage = 10;
    public array $perPageOptions = [10, 25, 50];

    // جزئیات و Quick View
    public bool $showQuickView = false;
    public $quickViewUser = null;

    protected $queryString = [
        'search' => ['except' => ''],
        'filterRole' => ['except' => ''],
        'filterStatus' => ['except' => ''],
        'filterCompanyId' => ['except' => null],
        'filterDepartmentId' => ['except' => null],
        'sortField' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'desc'],
        'perPage' => ['except' => 10],
    ];

    public function mount()
    {
        if (Gate::denies('manage-structure')) {
            abort(403, 'شما مجوز مدیریت کاربران را ندارید.');
        }

        $this->departments = $this->company_id
            ? Department::where('company_id', $this->company_id)->orderBy('name')->get()
            : [];
        // در حالت ویرایش اگر کاربر از طریق URL آمده باشد
        if ($this->userId) {
            $this->editUser($this->userId);
        }
    }

    public function updatedCompanyId($company_id)
    {
        $this->department_id = null;
        $this->departments = $company_id
            ? Department::where('company_id', $company_id)->orderBy('name')->get()
            : [];
        $this->resetValidation(['department_id']);
    }

    public function updatedFilterCompanyId()
    {
        $this->filterDepartmentId = null;
        $this->resetPage();
    }

    public function updatedSearch() { $this->resetPage(); }
    public function updatedFilterRole() { $this->resetPage(); }
    public function updatedFilterStatus() { $this->resetPage(); }
    public function updatedFilterDepartmentId() { $this->resetPage(); }
    public function updatedPerPage() { $this->resetPage(); }

    public function sortBy(string $field): void
    {
        if (!in_array($field, $this->sortable, true)) return;
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    // 💡 متد کمکی برای دریافت URL صحیح تصویر و مدیریت تصاویر پیش‌فرض
    public function getImageUrl(string $path = null, bool $isSign = false): string
    {
        // مسیر پیش‌فرض برای پروفایل و امضا
        $defaultProfile = asset('panel/dist/images/profile/user-1.jpg');
        $defaultSign = asset('panel/dist/images/default-sign.png'); // فرض کنید این فایل را دارید

        if ($path && Storage::disk('public')->exists($path)) {
            // اگر فایل در مسیر storage/app/public/users/... ذخیره شده باشد
            return Storage::url($path);
        }
        return $isSign ? $defaultSign : $defaultProfile;
    }

    public function openQuickView(int $userId): void
    {
        $this->quickViewUser = User::with([
            'department:id,name,company_id',
            'department.company:id,name',
        ])->findOrFail($userId);
        $this->showQuickView = true;
    }

    public function closeQuickView(): void
    {
        $this->showQuickView = false;
        $this->quickViewUser = null;
    }

    protected function rules()
    {
        return [
            'name'   => ['required','string','min:3','max:100'],
            'email'  => ['required','email', Rule::unique('users','email')->ignore($this->userId)],
            'mobile' => ['required','regex:/^09\d{9}$/', Rule::unique('users','mobile')->ignore($this->userId)],
            'ncode'  => ['required','digits:10', Rule::unique('users','ncode')->ignore($this->userId)],
            'internal_number' => ['nullable','string','max:20'],
            'role'   => ['required', Rule::in($this->roles)],
            'gender' => ['required', Rule::in(['male','female'])],
            'status' => ['required', Rule::in($this->statuses)],
            'degree' => ['nullable', Rule::in($this->degrees)],
            'company_id'   => ['nullable','exists:companies,id'],
            'department_id'=> [
                'nullable',
                Rule::exists('departments','id')->where(function($q){
                    if ($this->company_id) $q->where('company_id', $this->company_id);
                }),
            ],
            'image'  => ['nullable','image','mimes:jpg,jpeg,png,webp','max:2048'],
            'sign'   => ['nullable','image','mimes:jpg,jpeg,png,webp','max:2048'],
            'password' => $this->editMode ? ['nullable','min:6','confirmed'] : ['required','min:6','confirmed'],
        ];
    }

    protected function passwordResetRules()
    {
        return [
            'newPassword' => ['required','min:6'],
            'newPasswordConfirmation' => ['required','same:newPassword'],
        ];
    }

    protected $messages = [
        'mobile.regex' => 'شماره موبایل معتبر نیست (باید با 09 شروع و 11 رقمی باشد).',
        'department_id.exists' => 'دپارتمان معتبر نیست یا با شرکت انتخاب‌شده هم‌خوانی ندارد.',
        'password.confirmed' => 'تکرار رمز عبور با رمز جدید یکسان نیست.',
    ];

    public function render()
    {
        // گارد سورت در برابر دستکاری qs
        if (!in_array($this->sortField, $this->sortable, true)) {
            $this->sortField = 'created_at';
            $this->sortDirection = 'desc';
        }

        // آمار
        $totalUsers    = User::count();
        $activeCount   = User::where('status','active')->count();
        $inactiveCount = User::where('status','inactive')->count();

        // لیست
        $users = User::query()
            ->select(['id','name','email','mobile','internal_number','image','role','status','department_id','company_id','created_at','updated_at','last_login_at'])
            ->when($this->search, function($query) {
                $s = trim($this->search);
                $query->where(function($q) use ($s) {
                    $q->where('name','like',"%{$s}%")
                        ->orWhere('email','like',"%{$s}%")
                        ->orWhere('mobile','like',"%{$s}%")
                        ->orWhere('internal_number','like',"%{$s}%");
                });
            })
            ->when($this->filterRole, fn($q) => $q->where('role', $this->filterRole))
            ->when($this->filterStatus, fn($q) => $q->where('status', $this->filterStatus))
            ->when($this->filterCompanyId, fn($q) => $q->whereHas('department', fn($d) => $d->where('company_id', $this->filterCompanyId)))
            ->when($this->filterDepartmentId, fn($q) => $q->where('department_id', $this->filterDepartmentId))
            ->with([
                'department:id,name,company_id',
                'department.company:id,name',
            ]);

        // سورت با NULL-last برای last_login_at
        if ($this->sortField === 'last_login_at') {
            $dir = $this->sortDirection === 'asc' ? 'ASC' : 'DESC';
            $users->orderByRaw("last_login_at IS NULL, last_login_at {$dir}");
        } else {
            $users->orderBy($this->sortField, $this->sortDirection);
        }

        $users = $users->paginate($this->perPage);

        $companies = Company::orderBy('name')->get();
        $filterDepartments = $this->filterCompanyId
            ? Department::where('company_id',$this->filterCompanyId)->orderBy('name')->get()
            : collect();

        return view('livewire.admin.users.add-user', [
            'users' => $users,
            'companies' => $companies,
            'departments' => $this->departments,
            'degrees' => $this->degrees,
            'totalUsers' => $totalUsers,
            'activeCount' => $activeCount,
            'inactiveCount' => $inactiveCount,
            'filterDepartments' => $filterDepartments,
        ]);
    }

    public function resetListFilters(): void
    {
        $this->reset(['search','filterRole','filterStatus','filterCompanyId','filterDepartmentId','sortField','sortDirection']);
        $this->sortField = 'created_at';
        $this->sortDirection = 'desc';
        $this->resetPage();
    }

    public function editUser($userId)
    {
        $user = User::with('department')->findOrFail($userId);
        $this->editMode = true;
        $this->userId = $user->id;
        $this->name = $user->name;
        $this->email = $user->email;
        $this->mobile = $user->mobile;
        $this->ncode = $user->ncode;
        $this->internal_number = $user->internal_number;
        $this->role = $user->role;
        $this->gender = $user->gender;
        $this->status = $user->status;
        $this->degree = $user->degree;
        $this->company_id = $user->company_id ?? optional($user->department)->company_id;
        $this->departments = $this->company_id
            ? Department::where('company_id', $this->company_id)->orderBy('name')->get()
            : [];
        $this->department_id = $user->department_id;
        $this->currentImage = $user->image;
        $this->currentSign  = $user->sign;
        $this->resetValidation(['password','password_confirmation', 'image', 'sign']);
        // اسکرول به بالای صفحه یا فرم
        $this->dispatch('scroll-to-top');
    }

    public function saveUser()
    {
        $this->validate();
        $this->editMode ? $this->updateUser() : $this->addUser();
    }

    public function addUser()
    {
        try {
            $companyId = $this->company_id;
            if ($this->department_id) {
                $depCompanyId = Department::where('id',$this->department_id)->value('company_id');
                $companyId = $depCompanyId ?: $companyId;
            }

            User::create([
                'name'          => $this->name,
                'email'         => strtolower($this->email),
                'mobile'        => $this->mobile,
                'password'      => Hash::make($this->password),
                // 💡 استفاده از مسیردهی صحیح و نوع فایل
                'image'         => $this->uploadFile($this->image, 'public', null, 'profiles'),
                'sign'          => $this->uploadFile($this->sign, 'public', null, 'signs'),
                'ncode'         => $this->ncode,
                'internal_number' => $this->internal_number,
                'degree'        => $this->degree,
                'status'        => $this->status,
                'gender'        => $this->gender,
                'role'          => $this->role,
                'department_id' => $this->department_id,
                'company_id'    => $companyId,
                'is_active'     => $this->status === 'active',
            ]);

            $this->resetForm();
            session()->flash('success', 'کاربر جدید با موفقیت ایجاد شد.');
            $this->dispatch('user-saved');
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در ایجاد کاربر: ' . $e->getMessage());
        }
    }

    public function updateUser()
    {
        try {
            $user = User::findOrFail($this->userId);
            $companyId = $this->company_id;
            if ($this->department_id) {
                $depCompanyId = Department::where('id',$this->department_id)->value('company_id');
                $companyId = $depCompanyId ?: $this->company_id;
            }

            $data = [
                'name'          => $this->name,
                'email'         => strtolower($this->email),
                'mobile'        => $this->mobile,
                'ncode'         => $this->ncode,
                'internal_number' => $this->internal_number,
                'degree'        => $this->degree,
                'status'        => $this->status,
                'gender'        => $this->gender,
                'role'          => $this->role,
                'department_id' => $this->department_id,
                'company_id'    => $companyId,
                // 💡 استفاده از مسیردهی صحیح و نوع فایل
                'image'         => $this->image ? $this->uploadFile($this->image, 'public', $this->currentImage, 'profiles') : $this->currentImage,
                'sign'          => $this->sign ? $this->uploadFile($this->sign, 'public', $this->currentSign, 'signs') : $this->currentSign,
                'is_active'     => $this->status === 'active',
            ];

            if ($this->password) {
                $data['password'] = Hash::make($this->password);
            }

            $user->update($data);
            $this->resetForm();
            session()->flash('success', 'اطلاعات کاربر با موفقیت به‌روزرسانی شد.');
            $this->dispatch('user-saved');
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در به‌روزرسانی کاربر: ' . $e->getMessage());
        }
    }

    public function deleteCurrentFile(string $type): void
    {
        $path = $type === 'image' ? $this->currentImage : $this->currentSign;
        if ($path) {
            Storage::disk('public')->delete($path);
            if ($type === 'image') {
                $this->currentImage = null;
                $this->image = null;
            } else {
                $this->currentSign = null;
                $this->sign = null;
            }
        }
    }

    public function deleteUser($userId)
    {
        // برای نمایش نام کاربر در SweetAlert
        $user = User::select('name')->find($userId);
        $userName = $user ? $user->name : 'انتخاب شده';

        $this->dispatch('confirm-delete', [
            'userId' => $userId,
            'userName' => $userName,
        ]);
    }

    public function deleteConfirmed($userId)
    {
        try {
            $user = User::findOrFail($userId);
            if ($user->id === Auth::id()) {
                session()->flash('error', 'امکان حذف حساب کاربری خودتان وجود ندارد.');
                return;
            }
            if ($user->image) Storage::disk('public')->delete($user->image);
            if ($user->sign) Storage::disk('public')->delete($user->sign);
            $user->delete();
            session()->flash('success', 'کاربر با موفقیت حذف شد.');
            $this->resetPage();
            // اگر QuickView باز بود، آن را ببندد
            $this->closeQuickView();
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در حذف کاربر: ' . $e->getMessage());
        }
    }

    // Modal تغییر رمز
    public function showResetPasswordForm($userId)
    {
        $this->resetPasswordUserId = $userId;
        $this->quickViewUser = User::select('id', 'name')->find($userId);
        $this->newPassword = '';
        $this->newPasswordConfirmation = '';
        $this->resetValidation(['newPassword','newPasswordConfirmation']);
    }

    public function closeResetPasswordModal()
    {
        $this->reset(['resetPasswordUserId','newPassword','newPasswordConfirmation', 'quickViewUser']);
        $this->resetValidation(['newPassword','newPasswordConfirmation']);
    }

    public function updatePassword()
    {
        $this->validate($this->passwordResetRules(), [
            'newPassword.required' => 'رمز عبور جدید الزامی است.',
            'newPassword.min' => 'حداقل 6 کاراکتر.',
            'newPasswordConfirmation.required' => 'تکرار رمز الزامی است.',
            'newPasswordConfirmation.same' => 'تکرار با رمز جدید یکسان نیست.',
        ]);
        try {
            $user = User::findOrFail($this->resetPasswordUserId);
            $user->update(['password' => Hash::make($this->newPassword)]);
            session()->flash('success', 'رمز عبور کاربر با موفقیت تغییر کرد.');
            $this->closeQuickView(); // برای اطمینان از بسته شدن Quick View
            $this->closeResetPasswordModal();
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در تغییر رمز: ' . $e->getMessage());
        }
    }

    public function cancelEdit()
    {
        $this->resetForm();
        $this->editMode = false;
        $this->dispatch('scroll-to-top');
    }

    public function resetForm()
    {
        $this->reset([
            'name','email','mobile','password','password_confirmation',
            'ncode','internal_number','degree','role','gender','status',
            'company_id','department_id','departments',
            'image','sign','userId','editMode','currentImage','currentSign',
        ]);
        $this->role = 'user';
        $this->gender = 'male';
        $this->status = 'active';
        $this->departments = $this->company_id
            ? Department::where('company_id',$this->company_id)->orderBy('name')->get()
            : [];
        $this->resetValidation();
    }

    // 💡 متد آپلود بهبود یافته با مسیردهی
    protected function uploadFile($file, string $disk, string $oldFile = null, string $type = 'profiles'): ?string
    {
        if (!$file) return $oldFile;

        // مسیردهی به پوشه‌های اختصاصی
        $path = $type === 'signs' ? 'users/signs' : 'users/profiles';

        // حذف فایل قدیمی اگر وجود داشته باشد
        if ($oldFile && Storage::disk($disk)->exists($oldFile)) {
            Storage::disk($disk)->delete($oldFile);
        }

        // ذخیره فایل با نام تصادفی در پوشه مشخص
        return $file->store($path, $disk);
    }
}
