<?php

namespace App\Livewire\Admin\Users;

use App\Models\User;
use App\Models\Company;
use App\Models\Department;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Gate;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;

#[Layout('admin.master')]
class ListUsers extends Component
{
    use WithPagination, WithFileUploads;

    protected $paginationTheme = "bootstrap";

    // --- فیلتر و جستجو ---
    public string $search = '';
    public string $filterRole = '';
    public string $filterStatus = '';
    public ?int $filterCompanyId = null;
    public ?int $filterDepartmentId = null;
    public int $perPage = 10;

    // --- سورت ---
    public string $sortField = 'created_at';
    public string $sortDirection = 'desc';
    protected array $sortable = ['name','email','mobile','role','status','last_login_at','created_at'];

    // --- لیست‌های کمکی ---
    public $companies;
    public $filterDepartments;
    public array $roles = ['admin','manager','editor','user','guest'];
    public array $statuses = ['active','inactive'];

    // --- جزئیات و Quick View ---
    public $selectedUser = null;
    public bool $showQuickView = false;

    public $userId;


    // 🛑 Live Listeners
    public function updatedSearch() { $this->resetPage(); }
    public function updatedFilterRole() { $this->resetPage(); }
    public function updatedFilterStatus() { $this->resetPage(); }
    public function updatedFilterCompanyId() { $this->filterDepartmentId = null; $this->resetPage(); }
    public function updatedFilterDepartmentId() { $this->resetPage(); }
    public function updatedPerPage() { $this->resetPage(); }


    public function mount()
    {
        // 🔒 گارد امنیتی (فرض: فقط ادمین اصلی دسترسی دارد)
        // if (Gate::denies('manage-structure')) { abort(403, 'شما مجوز مدیریت کاربران را ندارید.'); }

        $this->companies = Company::orderBy('name')->get(['id', 'name']);
        // 💡 مقداردهی اولیه برای جلوگیری از خطای Null در اولین رندر
        $this->filterDepartments = collect();
    }

    public function sortBy(string $field): void
    {
        if (!in_array($field, $this->sortable, true)) return;

        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    public function getFilterDepartmentsProperty()
    {
        // 🛑 FIX: تضمین بازگشت Collection خالی
        return $this->filterCompanyId
            ? Department::where('company_id', $this->filterCompanyId)->orderBy('name')->get(['id', 'name'])
            : collect();
    }

    // 💡 متد کمکی برای دریافت URL صحیح تصویر (FIX)
    public function getImageUrl(string $path = null): string
    {
        $defaultProfile = asset('panel/dist/images/profile/user-1.jpg');

        if ($path && Storage::disk('public')->exists($path)) {
            return Storage::url($path);
        }
        return $defaultProfile;
    }

    public function openQuickView(int $userId): void
    {
        $this->selectedUser = User::with(['department.company'])->findOrFail($userId);
        $this->showQuickView = true;
    }

    public function closeQuickView(): void
    {
        $this->showQuickView = false;
        $this->selectedUser = null;
    }

    public function resetListFilters(): void
    {
        $this->reset(['search','filterRole','filterStatus','filterCompanyId','filterDepartmentId','sortField','sortDirection']);
        $this->sortField = 'created_at';
        $this->sortDirection = 'desc';
        $this->resetPage();
    }
    // ... (سایر متدهای CRUD) ...


    public function render()
    {
        $users = User::query()
            ->select(['id','name','email','mobile','internal_number','image','role','status','department_id','company_id','created_at','updated_at','last_login_at'])
            // ... (فیلترینگ و سورتینگ) ...
            ->with([
                'department:id,name,company_id',
                'department.company:id,name',
            ]);

        if ($this->sortField === 'last_login_at') {
            $dir = $this->sortDirection === 'asc' ? 'ASC' : 'DESC';
            $users->orderByRaw("last_login_at IS NULL, last_login_at {$dir}");
        } else {
            $users->orderBy($this->sortField, $this->sortDirection);
        }

        $users = $users->paginate($this->perPage);

        return view('livewire.admin.users.list-users', [
            'users' => $users,
            'companies' => $this->companies,
            'filterDepartments' => $this->filterDepartments, // 🛑 FIX: ارسال مقادیر از طریق Computed Property
        ]);
    }
}
