<?php


namespace App\Livewire\Admin\Users\Positions;

use App\Models\User;
use App\Models\UserPosition;
use App\Models\Company;
use App\Models\Department;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Gate;

class AddPosition extends Component
{
    // 🛑 USER ID را برای ذخیره نهایی نگه می‌داریم
    #[Validate('required|exists:users,id', message: 'لطفاً یک کاربر انتخاب کنید')]
    public $user_id = '';

    #[Validate('required|string|max:100', message: 'لطفاً سمت کاربر را وارد کنید')]
    public $position = '';

    public $searchTerm = '';

    // 🛑 NEW FILTER PROPERTIES
    public $companyIdFilter = '';
    public $departmentIdFilter = '';
    public $companies = [];
    public $departments = [];

    public $filteredUsers = [];
    public $selectedUserObject = null;
    public $isUserSelected = false;

    public function mount()
    {
        // 🛑 گارد امنیتی: فرض می‌کنیم Gate در AuthServiceProvider تعریف شده است
        if (Gate::denies('manage-positions')) {
            abort(403, 'شما مجوز مدیریت جایگاه‌های سازمانی را ندارید.');
        }

        // لود لیست‌های اصلی
        $this->companies = Company::select('id', 'name')->orderBy('name')->get();
        $this->refreshFilteredUsers();
        $this->loadDepartments();
    }

    // 💡 متد لود دپارتمان‌ها بر اساس شرکت انتخاب شده
    public function loadDepartments()
    {
        $query = Department::select('id', 'name');
        if ($this->companyIdFilter) {
            $query->where('company_id', $this->companyIdFilter);
        } else {
            $query->where('id', 0);
        }
        $this->departments = $query->orderBy('name')->get();
        $this->departmentIdFilter = '';
        $this->refreshFilteredUsers();
    }

    // 💡 متد اصلی برای فیلتر و جستجو
    public function refreshFilteredUsers()
    {
        $query = User::query()->select('id', 'name', 'email', 'image', 'department_id')
            ->with('department.company');

        // 1. جستجوی متنی
        if (strlen($this->searchTerm) >= 2 && !$this->isUserSelected) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('email', 'like', '%' . $this->searchTerm . '%');
            });
        } elseif (!$this->isUserSelected) {
            // در حالت عادی و بدون جستجو، لیست خالی باشد یا 10 نفر اول را نشان دهد
            $query->limit(10);
        }

        // 2. فیلتر شرکت (بر اساس دپارتمان)
        if ($this->companyIdFilter) {
            $query->whereHas('department', function ($q) {
                $q->where('company_id', $this->companyIdFilter);
            });
        }

        // 3. فیلتر دپارتمان
        if ($this->departmentIdFilter) {
            $query->where('department_id', $this->departmentIdFilter);
        }

        $this->filteredUsers = $query->orderBy('name')->limit(15)->get();
    }

    // 🛑 Live Listeners
    public function updatedSearchTerm()
    {
        $this->refreshFilteredUsers();
    }

    public function updatedCompanyIdFilter()
    {
        $this->loadDepartments();
    }

    public function updatedDepartmentIdFilter()
    {
        $this->refreshFilteredUsers();
    }

    // 💡 متد انتخاب کاربر از لیست جستجو
    public function selectUser(int $userId)
    {
        $user = User::with('department.company')->select('id', 'name', 'email', 'image', 'department_id')->find($userId);
        if ($user) {
            $this->user_id = $userId;
            $this->selectedUserObject = $user;
            $this->isUserSelected = true;
            $this->searchTerm = $user->name;
        }
        $this->resetValidation('user_id');
    }

    public function updated($propertyName)
    {
        if ($propertyName !== 'searchTerm') {
            $this->validateOnly($propertyName);
        }
    }

    public function saveUserPosition()
    {
        // 🛑 گارد مجدد
        if (Gate::denies('manage-positions')) {
            return;
        }

        $validated = $this->validate();

        try {
            // 🛑 FIX: چک کردن تکرار قبل از ثبت (برای جلوگیری از تکرار سوابق یکسان)
            $existingPosition = \App\Models\UserPosition::where('user_id', $validated['user_id'])
                ->where('position', $validated['position'])
                ->first();
            if ($existingPosition) {
                session()->flash('error', "این کاربر هم‌اکنون دارای سمت \"{$validated['position']}\" است.");
                return;
            }

            // 🛑 FIX: استفاده از CREATE برای ثبت چند سمت
            \App\Models\UserPosition::create([
                'user_id' => $validated['user_id'],
                'position' => $validated['position']
            ]);

            $userName = $this->selectedUserObject ? $this->selectedUserObject->name : 'کاربر';

            session()->flash('success', "سمت \"{$validated['position']}\" برای {$userName} با موفقیت ثبت شد.");
            $this->resetFields();
        } catch (\Exception $e) {
            session()->flash('error', 'خطا در ذخیره‌سازی اطلاعات. لطفاً دوباره تلاش کنید.');
        }
    }

    public function resetFields()
    {
        $this->reset(['user_id', 'position', 'searchTerm', 'selectedUserObject', 'isUserSelected', 'companyIdFilter', 'departmentIdFilter']);
        $this->resetValidation();
        $this->mount();
    }

    #[Layout('admin.master')]
    #[Title('افزودن سمت کاربر')]
    public function render()
    {
        return view('livewire.admin.users.positions.add-position');
    }
}



//
//namespace App\Livewire\Admin\Users\Positions;
//
//use App\Models\User;
//use App\Models\UserPosition;
//use App\Models\Company;
//use App\Models\Department;
//use Livewire\Attributes\Layout;
//use Livewire\Attributes\Title;
//use Livewire\Attributes\Validate;
//use Livewire\Component;
//use Illuminate\Support\Facades\DB;
//use Illuminate\Support\Str;
//
//class AddPosition extends Component
//{
//    // 🛑 USER ID را برای ذخیره نهایی نگه می‌داریم
//    #[Validate('required|exists:users,id', message: 'لطفاً یک کاربر انتخاب کنید')]
//    public $user_id = '';
//
//    #[Validate('required|string|max:100', message: 'لطفاً سمت کاربر را وارد کنید')]
//    public $position = '';
//
//    public $searchTerm = '';
//
//    // 🛑 NEW FILTER PROPERTIES
//    public $companyIdFilter = '';
//    public $departmentIdFilter = '';
//    public $companies = [];
//    public $departments = [];
//
//    public $filteredUsers = [];
//    public $selectedUserObject = null;
//    public $isUserSelected = false;
//
//    public function mount()
//    {
//        // لود لیست‌های اصلی
//        $this->companies = Company::select('id', 'name')->orderBy('name')->get();
//        $this->refreshFilteredUsers();
//        $this->loadDepartments();
//    }
//
//    // 💡 متد لود دپارتمان‌ها بر اساس شرکت انتخاب شده
//    public function loadDepartments()
//    {
//        $query = Department::select('id', 'name');
//        if ($this->companyIdFilter) {
//            $query->where('company_id', $this->companyIdFilter);
//        } else {
//            $query->where('id', 0);
//        }
//        $this->departments = $query->orderBy('name')->get();
//        $this->departmentIdFilter = '';
//    }
//
//    // 💡 متد اصلی برای فیلتر و جستجو
//    public function refreshFilteredUsers()
//    {
//        $query = User::query()->select('id', 'name', 'email', 'image', 'department_id')
//            ->with('department:id,company_id');
//
//        // 1. جستجوی متنی
//        if (strlen($this->searchTerm) >= 2 && !$this->isUserSelected) {
//            $query->where(function($q) {
//                $q->where('name', 'like', '%' . $this->searchTerm . '%')
//                    ->orWhere('email', 'like', '%' . $this->searchTerm . '%');
//            });
//        } elseif (!$this->isUserSelected) {
//            // در حالت عادی و بدون جستجو، لیست خالی باشد یا 10 نفر اول را نشان دهد
//            $query->limit(10);
//        }
//
//        // 2. فیلتر شرکت (بر اساس دپارتمان)
//        if ($this->companyIdFilter) {
//            $query->whereHas('department', function($q) {
//                $q->where('company_id', $this->companyIdFilter);
//            });
//        }
//
//        // 3. فیلتر دپارتمان
//        if ($this->departmentIdFilter) {
//            $query->where('department_id', $this->departmentIdFilter);
//        }
//
//        $this->filteredUsers = $query->orderBy('name')->limit(15)->get();
//    }
//
//    // 🛑 Live Listeners
//    public function updatedSearchTerm() { $this->refreshFilteredUsers(); }
//    public function updatedCompanyIdFilter() { $this->loadDepartments(); $this->refreshFilteredUsers(); }
//    public function updatedDepartmentIdFilter() { $this->refreshFilteredUsers(); }
//
//    // 💡 متد انتخاب کاربر از لیست جستجو
//    public function selectUser(int $userId)
//    {
//        $user = User::select('id', 'name', 'email', 'image')->find($userId);
//        if ($user) {
//            $this->user_id = $userId;
//            $this->selectedUserObject = $user;
//            $this->isUserSelected = true;
//            $this->searchTerm = $user->name; // نمایش نام در باکس جستجو
//        }
//        $this->resetValidation('user_id');
//    }
//
//    public function updated($propertyName)
//    {
//        if ($propertyName !== 'searchTerm') {
//            $this->validateOnly($propertyName);
//        }
//    }
//
//    public function saveUserPosition()
//    {
//        $validated = $this->validate();
//
//        try {
//            \App\Models\UserPosition::updateOrCreate(
//                ['user_id' => $validated['user_id']],
//                ['position' => $validated['position']]
//            );
//
//            $userName = $this->selectedUserObject ? $this->selectedUserObject->name : 'کاربر';
//
//            session()->flash('success', "سمت {$userName} با موفقیت به \"{$validated['position']}\" تغییر یافت.");
//            $this->resetFields();
//        } catch (\Exception $e) {
//            session()->flash('error', 'خطا در ذخیره‌سازی اطلاعات. لطفاً دوباره تلاش کنید.');
//        }
//    }
//
//    public function resetFields()
//    {
//        $this->reset(['user_id', 'position', 'searchTerm', 'selectedUserObject', 'isUserSelected', 'companyIdFilter', 'departmentIdFilter']);
//        $this->resetValidation();
//        $this->mount();
//    }
//
//    #[Layout('admin.master')]
//    #[Title('افزودن سمت کاربر')]
//    public function render()
//    {
//        return view('livewire.admin.users.positions.add-position');
//    }
//}
