<?php

namespace App\Livewire\Admin\Users\Positions;

use App\Models\UserPosition;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Gate;
use App\Models\Company;
use App\Models\Department;

class PositionList extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $search = '';
    public $perPage = 12;
    public $sortField = 'name';
    public $sortDirection = 'asc';

    public $companyIdFilter = '';
    public $departmentIdFilter = '';
    public $companies = [];
    public $departments = [];
    public $viewType = 'card';

    protected $queryString = [
        'search' => ['except' => ''],
        'sortField' => ['except' => 'name'],
        'sortDirection' => ['except' => 'asc'],
        'companyIdFilter' => ['except' => ''],
        'departmentIdFilter' => ['except' => ''],
        'viewType' => ['except' => 'card'],
    ];

    public function mount()
    {
        // 🔒 گارد امنیتی
        if (Gate::denies('manage-positions')) {
            abort(403, 'شما مجوز مدیریت جایگاه‌های سازمانی را ندارید.');
        }
        $this->companies = Company::select('id', 'name')->orderBy('name')->get();
        $this->loadDepartments();
    }

    public function toggleView()
    {
        $this->viewType = $this->viewType === 'table' ? 'card' : 'table';
    }

    public function loadDepartments()
    {
        $query = Department::select('id', 'name');
        if ($this->companyIdFilter) {
            $query->where('company_id', $this->companyIdFilter);
        } else {
            $query->where('id', 0);
        }
        $this->departments = $query->orderBy('name')->get();
        $this->departmentIdFilter = '';
        $this->resetPage();
    }

    public function updatedSearch() { $this->resetPage(); }
    public function updatedPerPage() { $this->resetPage(); }
    public function updatedCompanyIdFilter() { $this->loadDepartments(); }
    public function updatedDepartmentIdFilter() { $this->resetPage(); }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function deletePosition($id)
    {
        if (Gate::denies('manage-positions')) {
            session()->flash('error', 'شما مجوز حذف جایگاه‌ها را ندارید.');
            return;
        }

        $position = \App\Models\UserPosition::findOrFail($id);
        $userName = $position->user->name ?? 'کاربر';

        $position->delete();

        session()->flash('success', "سمت {$userName} با موفقیت حذف شد.");
    }

    #[Layout('admin.master')]
    public function render()
    {
        $usersQuery = \App\Models\User::query()
            ->whereHas('positions')
            ->with(['positions' => fn($q) => $q->orderBy('created_at', 'desc'),
                'department.company'])
            ->when($this->search, function($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('email', 'like', '%' . $this->search . '%')
                    ->orWhereHas('positions', fn($q) => $q->where('position', 'like', '%' . $this->search . '%'));
            })
            // 🛑 اعمال فیلتر شرکت/دپارتمان
            ->when($this->companyIdFilter, function ($query) {
                $query->whereHas('department', fn($q) => $q->where('company_id', $this->companyIdFilter));
            })
            ->when($this->departmentIdFilter, function ($query) {
                $query->where('department_id', $this->departmentIdFilter);
            });

        $usersQuery->orderBy($this->sortField, $this->sortDirection);

        $usersWithPositions = $usersQuery->paginate($this->perPage);

        return view('livewire.admin.users.positions.position-list', [
            'usersWithPositions' => $usersWithPositions,
            'viewType' => $this->viewType,
            'totalCount' => $usersQuery->count()
        ]);
    }
}
