<?php

namespace App\Livewire\Forms;

use App\Models\UserLoginHistory; // اضافه کردن مدل UserLoginHistory
use Illuminate\Auth\Events\Lockout;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Session; // اضافه کردن Facade Session
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Validate;
use Livewire\Form;

class LoginForm extends Form
{
    #[Validate('required|string|email')]
    public string $email = '';

    #[Validate('required|string')]
    public string $password = '';

    #[Validate('boolean')]
    public bool $remember = false;

    /**
     * Attempt to authenticate the request's credentials.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function authenticate(): void
    {
        $this->ensureIsNotRateLimited();

        if (! Auth::attempt($this->only(['email', 'password']), $this->remember)) {
            RateLimiter::hit($this->throttleKey());

            throw ValidationException::withMessages([
                'form.email' => trans('auth.failed'),
            ]);
        }

        // ✅ اینجا بعد از ورود موفق، تاریخچه رو ذخیره می‌کنیم
        $user = Auth::user(); // کاربر وارد شده

        UserLoginHistory::create([
            'user_id' => $user->id,
            'login_at' => now(), // همین الان
            'ip_address' => request()->ip(), // گرفتن IP
            'user_agent' => request()->userAgent(), // گرفتن User Agent
            // 'mac_address' => null, // همانطور که گفتیم، گرفتن MAC از طریق مرورگر امکان‌پذیر نیست
            // 'metadata' => json_encode(['user_agent' => request()->userAgent()]), // اگر خواستی اطلاعات بیشتری ذخیره کن
        ]);

        // ممکنه بخوای آخرین زمان ورود کاربر رو هم در مدل User بروزرسانی کنی
        $user->updateLastLogin(); // اگر این متد رو در مدل User داری

        // ✅ اضافه شده: Regenerate session ID برای امنیت
        Session::regenerate(); // این خط از کامپوننت Volt به اینجا منتقل شد و جای درستی هست

        RateLimiter::clear($this->throttleKey());
    }

    /**
     * Ensure the authentication request is not rate limited.
     */
    protected function ensureIsNotRateLimited(): void
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout(request()));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        throw ValidationException::withMessages([
            'form.email' => trans('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the authentication rate limiting throttle key.
     */
    protected function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->email).'|'.request()->ip());
    }
}
