<?php

namespace App\Livewire\Layout;

use App\Models\Announcement;
use App\Models\User;
use Illuminate\Notifications\DatabaseNotification;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class NotificationsDropdown extends Component
{
    public array $notifications = [];
    public int $unreadCount = 0;

    // متغیرهای هوشمندسازی: بارگذاری تدریجی
    public int $limit = 8;
    public bool $hasMore = false;

    public function mount(): void
    {
        $this->loadNotifications();
    }

    // 🎯 متد اصلی بارگذاری اعلان‌ها (فراخوانی از mount و wire:poll)
    public function loadNotifications(): void
    {
        $user = Auth::user();
        if (! $user) {
            $this->notifications = [];
            $this->unreadCount = 0;
            $this->hasMore = false;
            return;
        }

        // 1. اعلان‌های عمومی (Announcement)
        $announcements = Announcement::query()
            ->join('announcement_recipients as ar', function ($join) use ($user) {
                $join->on('ar.announcement_id', '=', 'announcements.id')
                    ->where('ar.user_id', $user->id);
            })
            ->with(['sender:id,name'])
            ->orderByDesc('announcements.created_at')
            ->get(['announcements.*', 'ar.read_at'])
            ->map(function ($item) {
                return [
                    'id'      => 'ann_' . $item->id,
                    'is_read' => (bool) $item->read_at,
                    'type'    => 'announcement',
                    'title'   => 'اطلاعیه عمومی: ' . $item->title,
                    'message' => $item->content,
                    'icon'    => 'ti ti-bell-school',
                    'color'   => 'primary',
                    'time'    => $item->created_at,
                    'url'     => route('user.posts'),
                    'sender'  => $item->sender?->name ?? 'سیستم',
                ];
            });


        // 2. اعلان‌های سیستمی (Database Notifications)
        $rawNotifications = $user->notifications()->orderByDesc('created_at')->get();
        $systemNotifications = new Collection();

        foreach ($rawNotifications as $item) {
            if (! $item instanceof DatabaseNotification || ! $item->id) {
                continue;
            }

            $data = is_array($item->data) ? $item->data : (array) ($item->data ?? []);

            $icon  = 'ti ti-bell-ringing';
            $color = 'info';
            $title = 'اعلان سیستمی';
            $message = $data['message'] ?? 'وضعیت سیستم به‌روزرسانی شد.';
            $url   = $data['url'] ?? route('admin.dashboard');
            $senderName = $data['sender_name'] ?? $data['by_name'] ?? $data['requester_name'] ?? 'سیستم';

            // 🎯 ۱. منطق تشخیص اعلان‌های فعال‌سازی حساب
            if (str_contains($item->type, 'AccountActivationRequest')) {
                if ($user->role === User::ROLE_ADMIN) {
                    $icon  = 'ti ti-user-exclamation';
                    $color = 'danger';
                    $title = '⚠️ درخواست فعال‌سازی حساب جدید';
                    $message = "کاربر «{$data['user_name']}» (کد: {$data['user_id']}) درخواست فعال‌سازی ارسال کرده است.";
                    $url   = route('admin.users.show', $data['user_id'] ?? 0);
                    $senderName = 'درخواست فعال‌سازی';
                } else {
                    continue;
                }
            }
            // 📬 ۲. منطق تشخیص اعلان‌های نامه
            elseif (str_contains($item->type, 'LetterReferralSent')) {
                $icon  = 'ti ti-mail-forward';
                $color = 'primary';
                $title = '📬 ارجاع نامه جدید';
                $purposeText = $data['purpose'] ?? 'اقدام';
                $message = "نامه «{$data['letter_title']}» جهت «{$purposeText}» ارجاع شد.";
                $url   = $data['url'] ?? route('letters.received');
                $senderName = $data['sender_name'] ?? 'سیستم ارجاعات';
            }
            elseif (str_contains($item->type, 'ReferralActionTaken')) {
                $action = $data['action'] ?? 'updated';
                $title = match ($action) {
                    'completed' => '✅ اقدام تکمیل شد',
                    'correction' => '🔄 نامه برگشت داده شد',
                    default => 'ℹ️ وضعیت ارجاع به‌روز شد',
                };
                $color = match ($action) {
                    'completed' => 'success',
                    'correction' => 'warning',
                    default => 'info',
                };
                $icon  = match ($action) {
                    'completed' => 'ti ti-file-check',
                    'correction' => 'ti ti-undo',
                    default => 'ti ti-eye',
                };
                $message = $data['message'] ?? "نامه «{$data['letter_title']}» توسط «{$senderName}» به‌روز شد.";
                $url   = $data['url'] ?? route('letters.received');
            }
            // 💰 ۳. منطق تشخیص اعلان‌های صندوق (PettyCash)
            elseif (str_contains($item->type, 'PettyCashNeedsApproval')) {
                $icon  = 'ti ti-wallet';
                $color = 'warning';
                $title = 'نیاز به تأیید درخواست تنخواه';
                $message = $data['message'] ?? 'درخواست جدید تنخواه نیاز به تأیید شما دارد.';
                $url   = $data['url'] ?? route('admin.petty-cash.index');
            } elseif (str_contains($item->type, 'PettyCashStatusUpdated')) {
                $icon  = 'ti ti-file-check';
                $color = ($data['status'] ?? 'pending') === 'approved' ? 'success' : 'danger';
                $title = 'وضعیت درخواست تنخواه به‌روز شد';
                $message = $data['message'] ?? 'وضعیت درخواست تنخواه شما تغییر کرد.';
                $url   = $data['url'] ?? route('admin.petty-cash.index');
            }
            // 🛒 ۴. منطق تشخیص اعلان‌های درخواست خرید (Purchase Request)
            elseif (str_contains($item->type, 'PurchaseRequestCreated')) {
                $icon  = 'ti ti-shopping-cart-plus';
                $color = 'indigo';
                $title = '🛒 درخواست خرید جدید';

                // ایمن‌سازی متغیرها با استفاده از ??
                $reqId = $data['request_id'] ?? 'نامشخص';
                $projName = $data['project_name'] ?? 'نامعلوم';

                $message = "درخواست خرید PR-{$reqId} برای پروژه «{$projName}» ثبت شد.";
                $url   = route('admin.purchase_request');
                $senderName = $data['user_name'] ?? 'تدارکات';
            }
            elseif (str_contains($item->type, 'PurchaseRequestStatusUpdated')) {
                $status = $data['status'] ?? 'pending';
                $icon  = $status === 'approved' ? 'ti ti-package-import' : 'ti ti-shopping-cart-x';
                $color = $status === 'approved' ? 'success' : 'danger';
                $title = $status === 'approved' ? '✅ تأیید درخواست خرید' : '❌ رد درخواست خرید';

                $projName = $data['project_name'] ?? 'نامعلوم';
                $message = $data['message'] ?? "وضعیت درخواست خرید شما برای پروژه «{$projName}» به‌روز شد.";
                $url   = route('admin.purchase_request');
                $senderName = 'سیستم تدارکات';
            }

            // Push آیتم نهایی
            $systemNotifications->push([
                'id'      => 'notif_' . $item->id,
                'is_read' => (bool) $item->read_at,
                'type'    => 'system',
                'title'   => $title,
                'message' => $message,
                'icon'    => $icon,
                'color'   => $color,
                'time'    => $item->created_at ?? now(),
                'url'     => $url,
                'sender'  => $senderName,
            ]);
        }

        // 3. ادغام، مرتب‌سازی و اعمال محدودیت بارگذاری تدریجی
        $all = $announcements
            ->concat($systemNotifications)
            ->sortByDesc('time')
            ->values();

        // محاسبه تعداد کل خوانده نشده‌ها (برای Badge)
        $this->unreadCount = $all->filter(fn($item) => $item['is_read'] === false)->count();

        // اعمال محدودیت (Limit)
        $this->notifications = $all->take($this->limit)->all();

        // تعیین وضعیت مشاهده موارد بیشتر
        $this->hasMore = $all->count() > $this->limit;
    }

    // متد برای بارگذاری بیشتر (توسط دکمه "مشاهده بیشتر")
    public function increaseLimit(): void
    {
        $this->limit += 8;
        $this->loadNotifications();
    }

    // 🎯 متد علامت‌گذاری یک آیتم خاص
    public function markAsRead(string $id, string $type): void
    {
        $user = Auth::user();
        if (! $user) return;

        $originalId = substr($id, 5);

        if ($type === 'announcement') {
            $announcementId = intval($originalId);

            if ($announcementId > 0) {
                DB::table('announcement_recipients')
                    ->where('announcement_id', $announcementId)
                    ->where('user_id', $user->id)
                    ->whereNull('read_at')
                    ->update(['read_at' => now()]);
            }
        } elseif ($type === 'system') {
            $notificationId = (string) $originalId;

            if (strlen($notificationId) > 10) {
                DB::table('notifications')
                    ->where('id', $notificationId)
                    ->where('notifiable_id', $user->id)
                    ->where('notifiable_type', $user->getMorphClass())
                    ->whereNull('read_at')
                    ->update(['read_at' => now()]);
            }
        }

        $this->loadNotifications();
        $this->dispatch('notification-read'); // برای به‌روزرسانی سایر کامپوننت‌ها
    }

    // متد برای علامت‌گذاری همه اعلان‌ها به عنوان خوانده شده
    public function markAllAsRead(): void
    {
        $user = Auth::user();
        if (! $user) return;

        DB::table('announcement_recipients')
            ->where('user_id', $user->id)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        DB::table('notifications')
            ->where('notifiable_type', $user->getMorphClass())
            ->where('notifiable_id', $user->id)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        $this->loadNotifications();
        $this->dispatch('notification-read'); // برای به‌روزرسانی سایر کامپوننت‌ها
    }

    public function render()
    {
        return view('livewire.layout.notifications-dropdown');
    }
}
