<?php

namespace App\Livewire\User;

use App\Models\Announcement;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('admin.master')]
class Inbox extends Component
{
    use WithPagination;

    #[Url(as: 'q')]   public string $search = '';
    #[Url(as: 'only')]public string $only   = 'all';       // all | unread | read
    #[Url(as: 'sort')]public string $sort   = 'newest';    // newest | oldest | unread_first
    #[Url(as: 'pp')]  public int    $perPage= 10;

    public array $selected = [];
    public bool  $selectPage = false;
    public int $tick = 0; // برای رفرش کارت‌ها

    protected $queryString = ['search', 'only', 'sort', 'perPage'];

    public function updatedPerPage($val): void
    {
        $this->perPage = in_array((int)$val, [5,10,15,20,50], true) ? (int)$val : 10;
        $this->resetPage();
        $this->clearSelection();
    }

    public function updatedSearch(): void { $this->resetPage(); $this->clearSelection(); }
    public function updatedOnly(): void   { $this->resetPage(); $this->clearSelection(); }
    public function updatedSort(): void   { $this->resetPage(); $this->clearSelection(); }

    protected function baseQuery()
    {
        $uid = Auth::id();

        $q = Announcement::query()
            ->select('announcements.*', 'ar.read_at as pivot_read_at')
            ->join('announcement_recipients as ar', function ($j) use ($uid) {
                $j->on('ar.announcement_id', '=', 'announcements.id')
                    ->where('ar.user_id', '=', $uid);
            })
            ->with([
                'sender:id,name',
                'attachments:id,announcement_id,path,original_name'
            ]);

        if ($this->only === 'unread') {
            $q->whereNull('ar.read_at');
        } elseif ($this->only === 'read') {
            $q->whereNotNull('ar.read_at');
        }

        if (filled($this->search)) {
            $s = trim($this->search);
            $q->where(function ($qq) use ($s) {
                $qq->where('announcements.title', 'like', "%{$s}%")
                    ->orWhere('announcements.content', 'like', "%{$s}%");
            });
        }

        if ($this->sort === 'oldest') {
            $q->orderBy('announcements.created_at', 'asc');
        } elseif ($this->sort === 'unread_first') {
            $q->orderByRaw('ar.read_at IS NULL DESC')->orderBy('announcements.created_at','desc');
        } else {
            $q->orderBy('announcements.created_at','desc');
        }

        return $q;
    }

    protected function counts(): array
    {
        $uid = Auth::id();
        $total  = DB::table('announcement_recipients')->where('user_id',$uid)->count();
        $unread = DB::table('announcement_recipients')->where('user_id',$uid)->whereNull('read_at')->count();
        $read   = $total - $unread;
        return compact('total','unread','read');
    }

    protected function currentPageIds()
    {
        return $this->baseQuery()->pluck('announcements.id');
    }

    public function updatedSelectPage($val): void
    {
        if ($val) {
            $this->selected = $this->currentPageIds()->all();
        } else {
            $this->selected = [];
        }
    }

    public function clearSelection(): void
    {
        $this->selected = [];
        $this->selectPage = false;
    }

    public function markAsRead(int $id): void
    {
        try {
            DB::table('announcement_recipients')
                ->where('user_id', Auth::id())
                ->where('announcement_id', $id)
                ->update(['read_at' => now()]);
            $this->tick++;
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در علامت‌گذاری به عنوان خوانده.');
        }
    }

    public function markAsUnread(int $id): void
    {
        try {
            DB::table('announcement_recipients')
                ->where('user_id', Auth::id())
                ->where('announcement_id', $id)
                ->update(['read_at' => null]);
            $this->tick++;
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در علامت‌گذاری به عنوان نخوانده.');
        }
    }

    public function toggleRead(int $id): void
    {
        $uid = Auth::id();
        $curr = DB::table('announcement_recipients')->where('user_id',$uid)->where('announcement_id',$id)->value('read_at');
        $curr ? $this->markAsUnread($id) : $this->markAsRead($id);
    }

    public function markAllAsRead(): void
    {
        try {
            $uid = Auth::id();
            $q = DB::table('announcement_recipients')->where('user_id',$uid)->whereNull('read_at');

            if ($this->only === 'unread' || filled($this->search)) {
                $ids = $this->baseQuery()->pluck('announcements.id');
                if ($ids->isNotEmpty()) $q->whereIn('announcement_id', $ids);
            }

            $q->update(['read_at' => now()]);
            $this->clearSelection();
            $this->tick++;
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در علامت‌گذاری گروهی.');
        }
    }

    public function markSelectedAsRead(): void
    {
        if (empty($this->selected)) return;
        try {
            DB::table('announcement_recipients')
                ->where('user_id', Auth::id())
                ->whereIn('announcement_id', $this->selected)
                ->update(['read_at' => now()]);
            $this->clearSelection();
            $this->tick++;
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در علامت‌گذاری گروهی.');
        }
    }

    public function markSelectedAsUnread(): void
    {
        if (empty($this->selected)) return;
        try {
            DB::table('announcement_recipients')
                ->where('user_id', Auth::id())
                ->whereIn('announcement_id', $this->selected)
                ->update(['read_at' => null]);
            $this->clearSelection();
            $this->tick++;
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در بازگردانی گروهی به نخوانده.');
        }
    }

    public function clearFilters(): void
    {
        $this->search = '';
        $this->only   = 'all';
        $this->sort   = 'newest';
        $this->perPage= 10;
        $this->resetPage();
        $this->clearSelection();
    }

    public function render()
    {
        $announcements = $this->baseQuery()->paginate($this->perPage);
        $stats = $this->counts();

        return view('livewire.user.inbox', [
            'announcements' => $announcements,
            'stats'         => $stats,
            'tick'          => $this->tick,
        ]);
    }
}
