<?php

namespace App\Livewire\User;

use App\Models\UserLoginHistory;
use Livewire\Component;

class LoginHistory extends Component
{
    public $loginHistories;
    public $stats;
    public $chartData = [];
    public $viewMode = 'table';
    public $showModal = false;
    public $modalData = [];

    public function mount()
    {
        // 💡 کد اصلاح شده برای رفرش اجباری پس از لاگین
        if (session('just_logged_in')) {
            // پاک کردن سشن برای جلوگیری از حلقه رفرش بی‌پایان
            session()->forget('just_logged_in');

            // اجرای رفرش کامل صفحه با جاوا اسکریپت (بهترین روش برای Livewire 3)
            // این کار تضمین می‌کند که تمام اسکریپت‌ها دوباره بارگذاری شوند.
            $this->js('window.location.reload(true)');
            return;
        }

        $this->loadAllData();
    }

    // ... ادامه کدها (loadAllData، getChartData، toggleViewMode، openDetails) بدون تغییر ...

    public function loadAllData()
    {
        $userId = auth()->id();

        $this->loginHistories = UserLoginHistory::where('user_id', $userId)
            ->orderBy('login_at', 'desc')
            ->limit(30)
            ->get();

        // آمار
        $last24h = now()->subHours(24);
        $this->stats = [
            'last_24h' => UserLoginHistory::where('user_id', $userId)
                ->where('login_at', '>=', $last24h)
                ->count(),
            'unique_ips' => $this->loginHistories->pluck('ip_address')->unique()->count(),
            'unique_countries' => $this->loginHistories->pluck('location.country')->unique()->count(),
        ];

        // داده‌های نمودار (7 روز گذشته)
        $this->chartData = $this->getChartData($userId);
    }

    protected function getChartData($userId)
    {
        $labels = [];
        $data = [];

        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $shamsi = \Morilog\Jalali\Jalalian::fromCarbon($date)->format('Y/m/d');
            $count = UserLoginHistory::where('user_id', $userId)
                ->whereDate('login_at', $date->toDateString())
                ->count();

            $labels[] = $shamsi;
            $data[] = $count;
        }

        return [
            'labels' => $labels,
            'data' => $data,
        ];
    }

    public function toggleViewMode($mode)
    {
        $this->viewMode = $mode;
    }

    public function openDetails($id)
    {
        $history = UserLoginHistory::where('user_id', auth()->id())->findOrFail($id);
        $this->modalData = [
            'id' => $history->id,
            'ip' => $history->ip_address,
            'user_agent' => $history->user_agent,
            'login_at' => $history->login_at->format('Y-m-d H:i:s'),
            'shamsi' => \Morilog\Jalali\Jalalian::fromCarbon($history->login_at)->format('Y/m/d H:i:s'),
            'location' => $history->location,
            'device' => $history->device,
        ];
        $this->showModal = true;
    }

    public function render()
    {
        return view('livewire.user.login-history')->layout('admin.master');
    }
}
