<?php

namespace App\Livewire\User;

use App\Models\User;
use App\Models\Company;
use App\Models\Department;
use Illuminate\Support\Facades\Hash;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

class MultiStepRegister extends Component
{
    use WithFileUploads;

    public int $step = 1;

    #[Validate('required|string|min:3|max:100')]
    public $name;
    #[Validate('required|email|unique:users,email')]
    public $email;
    #[Validate('required|regex:/^09\d{9}$/|unique:users,mobile')]
    public $mobile;
    #[Validate('required|digits:10|unique:users,ncode')]
    public $ncode;
    #[Validate('nullable|string|in:دیپلم,کاردانی,کارشناسی,کارشناسی ارشد,دکترا,بدون مدرک')]
    public $degree;
    #[Validate('nullable|exists:companies,id')]
    public $company_id;
    #[Validate('nullable|exists:departments,id')]
    public $department_id;
    #[Validate('required|string|min:6|confirmed')]
    public $password;
    #[Validate('required|string|min:6')]
    public $password_confirmation;
    #[Validate('required|in:male,female')]
    public $gender = 'male';
    #[Validate('nullable|image|mimes:jpg,jpeg,png,webp|max:2048')]
    public $image;
    #[Validate('nullable|image|mimes:jpg,jpeg,png,webp|max:2048')]
    public $sign;

    public array $degrees = ['دیپلم', 'کاردانی', 'کارشناسی', 'کارشناسی ارشد', 'دکترا', 'بدون مدرک'];
    public $companies = [];
    public $departments = [];

    protected $messages = [
        'mobile.regex' => 'شماره موبایل باید با 09 شروع شود و 11 رقم باشد.',
        'ncode.digits' => 'کد ملی باید 10 رقمی باشد.',
        'email.unique' => 'این ایمیل قبلاً ثبت شده است.',
        'mobile.unique' => 'این شماره موبایل قبلاً ثبت شده است.',
        'ncode.unique' => 'این کد ملی قبلاً ثبت شده است.',
        'password.min' => 'رمز عبور باید حداقل 6 کاراکتر باشد.',
    ];

    public function mount()
    {
        $this->companies = Company::orderBy('name')->get();
        $this->departments = [];
        if ($this->company_id) {
            $this->departments = Department::where('company_id', $this->company_id)->orderBy('name')->get();
        }
    }

    public function updatedCompanyId($value)
    {
        $this->department_id = null;
        $this->departments = $value
            ? Department::where('company_id', $value)->orderBy('name')->get()
            : [];
    }

    public function nextStep()
    {
        $rules = $this->getStepRules();
        $this->validate($rules, $this->messages);

        $oldStep = $this->step;
        if ($this->step < 3) $this->step++;

        // فورس ریفرش دپارتمان هر بار که به مرحله دوم می‌روی
        if ($this->step == 2) {
            if ($this->company_id) {
                $this->updatedCompanyId($this->company_id);
            } else {
                $this->departments = [];
            }
        }
    }

    public function previousStep()
    {
        if ($this->step > 1) {
            $this->step--;
            // اگر برگشتی به مرحله دوم ریفرش دپارتمان انجام شود
            if ($this->step == 2) {
                if ($this->company_id) {
                    $this->updatedCompanyId($this->company_id);
                } else {
                    $this->departments = [];
                }
            }
        }
    }

    private function getStepRules()
    {
        return match($this->step) {
            1 => ['name'=> $this->rules()['name'], 'email'=>$this->rules()['email'], 'mobile'=>$this->rules()['mobile'], 'ncode'=>$this->rules()['ncode']],
            2 => ['degree'=>$this->rules()['degree'], 'company_id'=>$this->rules()['company_id'], 'department_id'=>$this->rules()['department_id'], 'image'=>$this->rules()['image'],'sign'=>$this->rules()['sign']],
            3 => ['password'=>$this->rules()['password'], 'password_confirmation'=>$this->rules()['password_confirmation'], 'gender'=>$this->rules()['gender']],
        };
    }

    public function rules()
    {
        return [
            'name' => 'required|string|min:3|max:100',
            'email' => 'required|email|unique:users,email',
            'mobile' => 'required|regex:/^09\d{9}$/|unique:users,mobile',
            'ncode' => 'required|digits:10|unique:users,ncode',
            'degree' => 'nullable|string|in:دیپلم,کاردانی,کارشناسی,کارشناسی ارشد,دکترا,بدون مدرک',
            'company_id' => 'nullable|exists:companies,id',
            'department_id' => 'nullable|exists:departments,id',
            'password' => 'required|string|min:6|confirmed',
            'password_confirmation' => 'required|string|min:6',
            'gender' => 'required|in:male,female',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'sign' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        ];
    }

    public function submitForm()
    {
        $this->validate($this->rules(), $this->messages);

        try {
            $companyId = $this->company_id;
            if ($this->department_id) {
                $depCompanyId = Department::where('id', $this->department_id)->value('company_id');
                $companyId = $depCompanyId ?: $this->company_id;
            }
            $imagePath = $this->image ? $this->image->store('users', 'public') : null;
            $signPath = $this->sign ? $this->sign->store('users', 'public') : null;

            User::create([
                'name' => $this->name,
                'email' => strtolower($this->email),
                'mobile' => $this->mobile,
                'ncode' => $this->ncode,
                'password' => Hash::make($this->password),
                'degree' => $this->degree,
                'gender' => $this->gender,
                'department_id' => $this->department_id,
                'company_id' => $companyId,
                'image' => $imagePath,
                'sign' => $signPath,
                'status' => 'pending',
                'is_active' => false,
            ]);

            $this->resetExcept('step', 'degrees', 'companies');
            $this->step = 1;

            session()->flash('success', 'ثبت‌نام با موفقیت انجام شد. منتظر تایید مدیر باشید.');
            return redirect()->route('login');
        } catch (\Exception $e) {
            session()->flash('error', 'خطا در ثبت‌نام: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.user.multi-step-register')->layout('layouts.guest');
    }
}
