<?php

namespace App\Livewire\User;

use App\Models\Post;
use App\Models\PostComment;
use App\Models\PostLike;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('admin.master')]
class PostShow extends Component
{
    use WithPagination;

    public Post $post;

    public bool $liked = false;
    public int $likesCount = 0;
    public int $commentsCount = 0;

    #[Rule('required|string|min:3|max:1000')]
    public string $commentBody = '';

    protected $paginationTheme = 'bootstrap-5';

    public function mount(Post $post): void
    {
        $this->post = $post->load(['category', 'author']);

        $this->likesCount = $this->post->likes()->count();
        $this->commentsCount = $this->post->comments()->count();

        if (Auth::check()) {
            $this->liked = $this->post
                ->likes()
                ->where('user_id', Auth::id())
                ->exists();
        }
    }

    public function toggleLike(): void
    {
        if (!Auth::check()) {
            session()->flash('error', 'برای لایک کردن باید وارد حساب کاربری شوید.');
            return;
        }

        $userId = Auth::id();

        if ($this->liked) {
            PostLike::where('post_id', $this->post->id)
                ->where('user_id', $userId)
                ->delete();

            $this->liked = false;
            $this->likesCount = max(0, $this->likesCount - 1);
        } else {
            PostLike::firstOrCreate([
                'post_id' => $this->post->id,
                'user_id' => $userId,
            ]);

            $this->liked = true;
            $this->likesCount++;
        }
    }

    public function addComment(): void
    {
        if (!Auth::check()) {
            session()->flash('error', 'برای ارسال دیدگاه باید وارد حساب کاربری شوید.');
            return;
        }

        $this->validateOnly('commentBody');

        PostComment::create([
            'post_id' => $this->post->id,
            'user_id' => Auth::id(),
            'body'    => $this->commentBody,
        ]);

        $this->reset('commentBody');

        $this->commentsCount++;

        // برگردوندن صفحه‌ی کامنت‌ها به اول
        $this->resetPage();

        session()->flash('success', 'دیدگاه شما با موفقیت ثبت شد.');
    }

    public function getCommentsProperty()
    {
        return PostComment::with('user')
            ->where('post_id', $this->post->id)
            ->latest()
            ->paginate(5);
    }

    public function render()
    {
        return view('livewire.user.post-show', [
            'comments' => $this->comments,
        ]);
    }
}
