<?php

namespace App\Livewire\User;

use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\Company;
use App\Models\Department;

class ProfileForm extends Component
{
    use WithFileUploads;

    public $name, $ncode, $email, $mobile, $role, $degree, $gender, $is_active, $internal_number;
    public $company_id, $department_id;
    public $image, $sign;
    public $image_preview, $sign_preview;

    // برای تغییر رمز عبور
    public $current_password, $new_password, $new_password_confirmation;

    // لیست‌های ثابت
    public array $degrees = [
        'دیپلم',
        'کاردانی',
        'کارشناسی',
        'کارشناسی ارشد',
        'دکترا',
        'بدون مدرک'
    ];

    // نقش‌ها به صورت key => label
    public array $roles = [
        'admin' => 'مدیر سیستم',
        'manager' => 'مدیر پروژه',
        'editor' => 'ویرایشگر',
        'user' => 'کاربر عادی',
        'guest' => 'مهمان'
    ];

    public function mount()
    {
        $user = Auth::user();
        $this->name   = $user->name;
        $this->ncode  = $user->ncode;
        $this->email  = $user->email;
        $this->mobile = $user->mobile;
        $this->role   = $user->role;
        $this->degree = $user->degree;
        $this->gender = $user->gender;
        $this->is_active = $user->is_active;
        $this->internal_number = $user->internal_number;
        $this->company_id = $user->company_id;
        $this->department_id = $user->department_id;

        $this->image_preview = $user->image ? asset('storage/'.$user->image) : null;
        $this->sign_preview  = $user->sign  ? asset('storage/'.$user->sign)  : null;
    }

    protected $rules = [
        'name'   => 'required|string|max:255',
        'ncode'  => 'required|string|max:20',
        'email'  => 'required|email',
        'mobile' => 'nullable|string|max:20',
        'role'   => 'nullable|string',
        'degree' => 'nullable|string',
        'gender' => 'nullable|in:male,female',
        'is_active' => 'boolean',
        'internal_number' => 'nullable|string|max:20',
        'company_id' => 'nullable|exists:companies,id',
        'department_id' => 'nullable|exists:departments,id',
        'image'  => 'nullable|image|max:2048',
        'sign'   => 'nullable|image|max:1024',
    ];

    protected $passwordRules = [
        'current_password' => 'required',
        'new_password' => 'required|min:8|confirmed',
    ];

    public function updatedImage()
    {
        $this->validateOnly('image');
        $this->image_preview = $this->image->temporaryUrl();
    }

    public function updatedSign()
    {
        $this->validateOnly('sign');
        $this->sign_preview = $this->sign->temporaryUrl();
    }

    public function updatedCompanyId($value)
    {
        $this->department_id = null;
    }

    public function save()
    {
        $this->validate();

        $user = Auth::user();

        $user->name   = $this->name;
        $user->ncode  = $this->ncode;
        $user->email  = $this->email;
        $user->mobile = $this->mobile;
        $user->role   = $this->role;
        $user->degree = $this->degree;
        $user->gender = $this->gender;
        $user->is_active = (bool) $this->is_active;
        $user->internal_number = $this->internal_number;
        $user->company_id = $this->company_id;
        $user->department_id = $this->department_id;

        if ($this->image) {
            $user->image = $this->image->store('users/images', 'public');
        }
        if ($this->sign) {
            $user->sign = $this->sign->store('users/signs', 'public');
        }

        $user->save();

        session()->flash('success', 'پروفایل با موفقیت بروزرسانی شد.');
        $this->mount(); // Refresh data
    }

    public function changePassword()
    {
        $this->validate($this->passwordRules);

        $user = Auth::user();

        if (!Hash::check($this->current_password, $user->password)) {
            $this->addError('current_password', 'رمز عبور فعلی صحیح نیست.');
            return;
        }

        $user->password = Hash::make($this->new_password);
        $user->save();

        session()->flash('success', 'رمز عبور با موفقیت تغییر کرد.');

        $this->current_password = $this->new_password = $this->new_password_confirmation = null;
    }

    #[Layout('admin.master')]
    public function render()
    {
        // 👇👇👇 داده‌ها را مستقیماً در اینجا محاسبه و ارسال کنید 👇👇👇
        $companies = Company::orderBy('name')->get();
        $departments = $this->company_id
            ? Department::where('company_id', $this->company_id)->orderBy('name')->get()
            : collect();

        return view('livewire.user.profile-form', [
            'companies' => $companies,
            'departments' => $departments,
        ]);
    }
}
