<?php

namespace App\Livewire\User;

use App\Models\Announcement;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('admin.master')]
class SentAnnouncements extends Component
{
    use WithPagination;

    protected string $paginationTheme = 'bootstrap-5';

    #[Url(as: 'q')]
    public string $search = '';

    #[Url(as: 'sort')]
    public string $sort = 'newest';

    #[Url(as: 'pp')]
    public int $perPage = 10;

    public ?int $expandedId = null;

    public function mount(): void
    {
        $this->perPage = $this->normalizePerPage($this->perPage);
    }

    protected function normalizePerPage(int $value): int
    {
        $allowed = [5, 10, 15, 20, 50];

        return in_array($value, $allowed, true) ? $value : 10;
    }

    public function updatedPerPage($val): void
    {
        $this->perPage = $this->normalizePerPage((int) $val);
        $this->resetPage();
        $this->expandedId = null;
    }

    public function updatedSearch(): void
    {
        $this->resetPage();
        $this->expandedId = null;
    }

    public function updatedSort(): void
    {
        $this->resetPage();
        $this->expandedId = null;
    }

    public function toggleExpansion(?int $id): void
    {
        $this->expandedId = $this->expandedId === $id ? null : $id;
    }

    public function deleteAnnouncement(int $id): void
    {
        try {
            $announcement = Announcement::where('sender_id', Auth::id())->findOrFail($id);
            $title = $announcement->title;
            $announcement->delete();

            session()->flash('success', "اطلاعیه «{$title}» با موفقیت حذف شد.");
        } catch (\Throwable $e) {
            session()->flash('error', 'خطا در حذف اطلاعیه. لطفاً دوباره تلاش کنید.');
        }
    }

    protected function baseQuery()
    {
        $uid = Auth::id();

        $announcements = Announcement::query()
            ->where('sender_id', $uid)
            ->with('attachments');

        if (filled($this->search)) {
            $s = trim($this->search);

            $announcements->where(function ($q) use ($s) {
                $q->where('title', 'like', "%{$s}%")
                    ->orWhere('content', 'like', "%{$s}%");
            });
        }

        $announcements->withCount([
            'recipients as read_count' => fn($query) => $query->whereNotNull('read_at'),
            'recipients as total_count',
        ]);

        $announcements->orderBy('created_at', $this->sort === 'oldest' ? 'asc' : 'desc');

        return $announcements;
    }

    protected function getExpandedRecipients()
    {
        if (!$this->expandedId) {
            return collect();
        }

        return DB::table('announcement_recipients as ar')
            ->join('users as u', 'u.id', '=', 'ar.user_id')
            ->where('ar.announcement_id', $this->expandedId)
            ->select('ar.*', 'u.name', 'u.email')
            ->orderBy('u.name')
            ->get();
    }

    public function render()
    {
        $announcements = $this->baseQuery()->paginate($this->perPage);
        $expandedRecipients = $this->getExpandedRecipients();

        return view('livewire.user.sent-announcements', [
            'announcements'      => $announcements,
            'expandedRecipients' => $expandedRecipients,
        ]);
    }
}
