<?php

namespace App\Livewire\User;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Post;
use Illuminate\Support\Facades\Auth;

class UserPosts extends Component
{
    use WithPagination;

    public $search = '';
    public $showForm = false;
    public $editId = null;
    public $title = '';
    public $content = '';
    public $mode = 'create';

    protected $paginationTheme = 'bootstrap';

    protected $rules = [
        'title' => 'required|string|max:255',
        'content' => 'required|string',
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function create()
    {
        $this->resetForm();
        $this->mode = 'create';
        $this->showForm = true;
    }

    public function edit($id)
    {
        $post = Post::findOrFail($id);
        if (Auth::id() !== $post->user_id && Auth::user()->role !== 'admin') {
            session()->flash('error', 'دسترسی ندارید.');
            return;
        }
        $this->editId = $post->id;
        $this->title = $post->title;
        $this->content = $post->content;
        $this->mode = 'edit';
        $this->showForm = true;
    }

    public function save()
    {
        $this->validate();

        if ($this->mode === 'edit' && $this->editId) {
            $post = Post::findOrFail($this->editId);
            if (Auth::id() !== $post->user_id && Auth::user()->role !== 'admin') {
                session()->flash('error', 'دسترسی ندارید.');
                return;
            }
            $post->update([
                'title' => $this->title,
                'content' => $this->content,
            ]);
            session()->flash('success', 'پست ویرایش شد.');
        } else {
            Post::create([
                'title' => $this->title,
                'content' => $this->content,
                'user_id' => Auth::id(),
            ]);
            session()->flash('success', 'پست جدید ثبت شد.');
        }

        $this->resetForm();
        $this->showForm = false;
    }

    public function delete($id)
    {
        $post = Post::findOrFail($id);
        if (Auth::id() === $post->user_id || Auth::user()->role === 'admin') {
            $post->delete();
            session()->flash('success', 'پست حذف شد.');
        } else {
            session()->flash('error', 'دسترسی ندارید.');
        }
    }

    public function resetForm()
    {
        $this->editId = null;
        $this->title = '';
        $this->content = '';
        $this->mode = 'create';
    }

    public function closeForm()
    {
        $this->showForm = false;
        $this->resetForm();
    }

    public function getPostsProperty()
    {
        // فقط پست‌های کاربر جاری
        return Post::with('user')
            ->where('user_id', Auth::id())
            ->where(function($query) {
                $query->where('title', 'like', "%{$this->search}%")
                    ->orWhere('content', 'like', "%{$this->search}%");
            })
            ->latest()
            ->paginate(8);
    }

    public function render()
    {
        return view('livewire.user.user-posts');
    }
}
